package com.xebialabs.deployit.cli.api;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;
import static java.lang.String.format;

import java.io.IOException;
import java.util.Arrays;
import java.util.Calendar;
import java.util.List;

import javax.ws.rs.core.Response;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import com.xebialabs.deployit.core.api.dto.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.api.internal.DescriptorHelper;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.ExportHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.cli.rest.ResponseExtractor;
import com.xebialabs.deployit.core.api.resteasy.Date;


@CliObject(name = "repository")
@ClassHelp(description = "Gateway to doing CRUD operations on all types of CIs")
public class RepositoryClient extends DocumentedObject {
	public static final Function<ConfigurationItemId,String> ciIdToString = new Function<ConfigurationItemId, String>() {
		public String apply(ConfigurationItemId input) {
			return input.getId();
		}
	};
	private final Proxies proxies;
	private final DescriptorHelper descriptors;
	private final String parseDateFormat = "MM/dd/yyyy";
	private final String resultDateFormat = "dd MMM yyyy HH:mm z";

	public RepositoryClient() {
		this.proxies = null;
		this.descriptors = null;
	}

	public RepositoryClient(final Proxies proxies) {
		this.proxies = proxies;
		this.descriptors = DescriptorHelper.getInstance(proxies);
	}

	@MethodHelp(description = "Create a new CI in the repository", parameters = {
	        @ParameterHelp(name = "ci", description = "The CI (ConfigurationItem) that should be created in the repository") })
	public RepositoryObject create(final ConfigurationItemDto object) {
		checkArgument(object.getId() != null, "The repository object should have a set id.");
        final Response response = proxies.getRepository().create(object.getId(), object);
        return checkForValidations(response);
	}

    @MethodHelp(description = "Create a new artifact CI in the repository", parameters = {
	        @ParameterHelp(name = "artifact", description = "The Artifact that should be created in the repository") })
	public RepositoryObject create(final ArtifactAndData artifact) {
	    checkArgument(artifact.getArtifact().getId() != null, "The artifact should have a set id.");
        final Response response = proxies.getRepository().create(artifact.getArtifact().getId(), artifact);
        return checkForValidations(response);
	}

	@MethodHelp(description = "Create all new CIs in the repository, commonly used after a discovery", parameters = {
	        @ParameterHelp(name = "cis", description = "The CIs (ConfigurationItems) that should be created in the repository") })
	public ConfigurationItemDtos create(final ConfigurationItemDtos repositoryObjects) {
		final Response response = proxies.getRepository().createMultiple(repositoryObjects);
		return checkAllForValidations(response);
	}

	@MethodHelp(description = "Create all new CIs in the repository, commonly used after a discovery", parameters = {
	        @ParameterHelp(name = "cis", description = "The CIs (ConfigurationItems) that should be created in the repository") })
	public ConfigurationItemDtos create(final List<ConfigurationItemDto> dtos) {
        final Response response = proxies.getRepository().createMultiple(new ConfigurationItemDtos(dtos));
		return checkAllForValidations(response);
	}

	private ConfigurationItemDtos checkAllForValidations(Response response) {
		final ResponseExtractor responseExtractor = new ResponseExtractor(response);
		final ConfigurationItemDtos cis = responseExtractor.getEntity();
		if (!responseExtractor.isValidResponse()) {
			for (RepositoryObject repositoryObject : cis.getObjects()) {
				if (!repositoryObject.getValidations().isEmpty()) {
					logger.error("Configuration item contained validation errors: {}", repositoryObject.getValidations());
				}
			}
		}
		return cis;
	}

	@MethodHelp(description = "Read a CI form the repository", parameters = { @ParameterHelp(name = "id", description = "The id of the CI to read") })
	public RepositoryObject read(final String id) {
		return new ResponseExtractor(proxies.getRepository().read(id)).getEntity();
	}

	@MethodHelp(description = "Update an existing CI in the repository", parameters = {
	        @ParameterHelp(name = "ci", description = "The updated CI (ConfigurationItem) that should be stored in the repository") })
	public RepositoryObject update(final ConfigurationItemDto object) {
		checkArgument(object.getId() != null, "The repository object should have a set id.");
        final Response response = proxies.getRepository().update(object.getId(), object);
        return checkForValidations(response);
	}

	@MethodHelp(description = "Update an existing artifact in the repository", parameters = {
	        @ParameterHelp(name = "artifact", description = "The updated artifact CI that should be stored in the repository") })
	public RepositoryObject update(final ArtifactAndData artifact) {
		checkArgument(artifact.getArtifact().getId() != null, "The repository object should have a set id.");
        final Response response = proxies.getRepository().update(artifact.getArtifact().getId(), artifact);
        return checkForValidations(response);
	}

	@MethodHelp(description = "Move a CI from one location to another", parameters = {
			@ParameterHelp(name = "ci", description = "The CI that is to be moved"),
			@ParameterHelp(name = "newId", description = "The new id of the CI")
	}, returns = "The CI with the new id")
	public RepositoryObject move(final ConfigurationItemDto ci, final String newId) {
		return move(ci.getId(), newId);
	}

	@MethodHelp(description = "Move a CI from one location to another", parameters = {
			@ParameterHelp(name = "id", description = "The id of the CI that is to be moved"),
			@ParameterHelp(name = "newId", description = "The new id of the CI")
	}, returns = "The CI with the new id")
	public RepositoryObject move(final String id, final String newId) {
		return new ResponseExtractor(proxies.getRepository().move(id, newId)).getEntity();
	}

	@MethodHelp(description = "Rename a CI", parameters = {
			@ParameterHelp(name = "ci", description = "The CI to rename"),
			@ParameterHelp(name = "newName", description = "The new name (last part of the id)")
	}, returns = "The CI with the updated name")
	public RepositoryObject rename(final ConfigurationItemDto ci, final String newName) {
		return rename(ci.getId(), newName);
	}

	@MethodHelp(description = "Rename a CI", parameters = {
			@ParameterHelp(name = "id", description = "The id of the CI to rename"),
			@ParameterHelp(name = "newName", description = "The new name (last part of the id)")
	}, returns = "The CI with the updated name")
	public RepositoryObject rename(String id, String newName) {
		return new ResponseExtractor(proxies.getRepository().rename(id, newName)).getEntity();
	}

	@MethodHelp(description = "Delete a CI with a specific id from the repository", parameters = { @ParameterHelp(name = "id", description = "The id of the CI") })
	public void delete(final String id) {
		new ResponseExtractor(proxies.getRepository().delete(id));
	}

	@MethodHelp(description = "Search for CIs of a specific type in the repository", parameters = { @ParameterHelp(name = "ciType", description = "") }, returns = "The ids of the configuration items that fit the query.")
	public List<String> search(String ciType) {
        String type = null;

        if (ciType != null) {
		    type = descriptors.getRegisteredType(ciType);
            if (type == null) {
                System.err.println("Configuration item type [null] not known.");
                return null;
            }
        }
		ConfigurationItemIds ids = new ResponseExtractor(proxies.getQuery().list(type, Boolean.FALSE, 0, -1, null, null, null)).getEntity();
		return newArrayList(Lists.transform(ids.getConfigurationItemIds(), ciIdToString));
	}

	@MethodHelp(description = "Search for CIs of a specific type in the repository which are located under the specified parent node.", parameters = {
			@ParameterHelp(name = "ciType", description = "The type of configuration item to look for (eg. udm.DeployedApplication)"),
			@ParameterHelp(name = "parent", description = "The id of the parent node to search under (eg. Environments/env1)") },
	returns = "The ids of the configuration items that fit the query.")
	public List<String> search(String ciType, String parent) {
		final String type = descriptors.getRegisteredType(ciType);
		ConfigurationItemIds ids = new ResponseExtractor(proxies.getQuery().list(type, Boolean.FALSE, 0, -1, null, parent, null)).getEntity();
		return newArrayList(Lists.transform(ids.getConfigurationItemIds(), ciIdToString));
	}

	@MethodHelp(description = "Search for CIs of a specific type in the repository were created before the given date.", parameters = {
			@ParameterHelp(name = "ciType", description = "The type of configuration item to look for (eg. udm.DeployedApplication)"),
			@ParameterHelp(name = "before", description = "a java.util.Calendar which specifies a moment in time before which the CI has to be created") },
			returns = "The ids of the configuration items that fit the query.")
	public List<String> search(String ciType, Calendar c) {
		ConfigurationItemIds ids = new ResponseExtractor(proxies.getQuery().search(ciType, new Date(c))).getEntity();
		return newArrayList(Lists.transform(ids.getConfigurationItemIds(), ciIdToString));
	}

	@MethodHelp(description = "Read multiple objects from the repostory in one go.", parameters = { @ParameterHelp(name = "ids", description = "The ids of the objects to read") })
	public List<ConfigurationItemDto> read(String... ids) {
		ConfigurationItemIds configurationItemIds = new ConfigurationItemIds();
		configurationItemIds.setConfigurationItemIds(Lists.transform(Arrays.asList(ids), new Function<String, ConfigurationItemId>() {
			public ConfigurationItemId apply(String input) {
				return new ConfigurationItemId(input, null);
			}
		}));
		final ConfigurationItemDtos objs = new ResponseExtractor(proxies.getQuery().readMultiple(configurationItemIds)).getEntity();
		return objs.getObjects();
	}

	@MethodHelp(description = "Get all task information, including steps, from the repository's archive.",
			returns = "This object contains all archived tasks with their enclosed steps")
	public FullTaskInfos getArchivedTasks() {
		return new ResponseExtractor(proxies.getTaskExport().getFullArchivedTasks()).getEntity();
	};

	@MethodHelp(description = "Get all task information, including steps, from the repository's archive in the specified date range.", parameters = {
			@ParameterHelp(name = "beginDate", description = "Begin date from which to return tasks in 'MM/dd/yyyy' format"),
			@ParameterHelp(name = "endDate", description = "End date from which to return tasks in 'MM/dd/yyyy' format")},
			returns = "This object contains all archived tasks with their enclosed steps")
	public FullTaskInfos getArchivedTasks(final String beginDate, final String endDate){
		return new ResponseExtractor(proxies.getTaskExport().getFullArchivedTasksInDateRange(new Date(ExportHelp.parseDateFormat(beginDate, parseDateFormat, resultDateFormat)),
				new Date(ExportHelp.parseDateFormat(endDate, parseDateFormat, resultDateFormat)))).getEntity();
	}

	@MethodHelp(description = "Export all task information, including steps, from the repository's archive to a local XML file", parameters = {
			@ParameterHelp(name = "filePath", description = "Fully qualified pathname, including the file name, as to where to store the file to. Example: '/tmp/exportedTasks.xml'")})
	public void exportArchivedTasks(final String filePath) throws IOException {
		getResourceToWrite(filePath, "/deployit/export/tasks");
	}

	@MethodHelp(description = "Export all task information, including steps, from the repository's archive in the specified date range to a local XML file", parameters = {
			@ParameterHelp(name = "filePath", description = "Fully qualified pathname, including the file name, as to where to store the file to. Example: '/tmp/exportedTasks.xml'"),
			@ParameterHelp(name = "beginDate", description = "Begin date from which to return tasks in 'MM/dd/yyyy' format"),
			@ParameterHelp(name = "endDate", description = "End date from which to return tasks in 'MM/dd/yyyy' format")})
	public void exportArchivedTasks(final String filePath, final String beginDate, final String endDate) throws IOException {
		getResourceToWrite(filePath, format("/deployit/export/taskrange?begin=%s&end=%s",
				ExportHelp.encodeURIDateFormat(ExportHelp.parseDateFormat(beginDate, parseDateFormat, resultDateFormat), "UTF-8"),
				ExportHelp.encodeURIDateFormat(ExportHelp.parseDateFormat(endDate, parseDateFormat, resultDateFormat), "UTF-8")));
	}

    @MethodHelp(description = "Check for existence of an id, does not guarantee read access to said configuration item.", parameters = {
            @ParameterHelp(name = "id", description = "The id of the configuration item")
    }, returns = "true if there is a configuration item with that id.")
    public boolean exists(String id) {
        ResponseExtractor responseExtractor = new ResponseExtractor(proxies.getRepository().exists(id));
        return responseExtractor.isValidResponse();
    }
    
	private RepositoryObject checkForValidations(final Response response) {
		final ResponseExtractor responseExtractor = new ResponseExtractor(response);
		final RepositoryObject ci = responseExtractor.getEntity();
		if (!responseExtractor.isValidResponse() && !ci.getValidations().isEmpty()) {
			logger.error("Configuration item contained validation errors: {}", ci.getValidations());
		}
		return ci;
	}

	private final void getResourceToWrite(final String filePath, final String resourcePath) throws IOException {
		if (ExportHelp.writeResourceToLocalFile(proxies.getAuthentication(), filePath, resourcePath) > 0) {
			System.out.println(format("Written archived tasks to %s", filePath));
		} else {
			System.out.println("File was created but no bytes were written to the file.\n");
			System.out.println("Maybe the requested resource was not available or zero size!?");
		}
	}

    private static final Logger logger = LoggerFactory.getLogger(RepositoryClient.class);
}
