package com.xebialabs.deployit.cli.api.internal;

import static com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyKind.ENUM;
import static java.lang.String.format;
import static java.lang.System.out;

import java.util.Arrays;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Sets;
import com.xebialabs.deployit.cli.api.Proxies;
import com.xebialabs.deployit.cli.rest.ResponseExtractor;
import com.xebialabs.deployit.core.api.ReferenceDataProxy;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemDescriptorList;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemMethodDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyDescriptorDto;
import com.xebialabs.deployit.core.api.dto.ConfigurationItemPropertyKind;

public class DescriptorHelper {
    private static volatile DescriptorHelper instance;

    private ReferenceDataProxy descriptorProxy;

    private Set<String> simpleNameMap = Sets.newHashSet();


    public DescriptorHelper(final Proxies proxies) {
        this.descriptorProxy = proxies.getReferenceData();
        initialize();
    }

    private void initialize() {
        final ConfigurationItemDescriptorList descriptors = new ResponseExtractor(descriptorProxy.list()).getEntity();
        registerSimpleNames(descriptors);
    }

    private void registerSimpleNames(final ConfigurationItemDescriptorList descriptors) {
        for (ConfigurationItemDescriptorDto configurationItemDescriptorDto : descriptors.getDescriptors()) {
            simpleNameMap.add(configurationItemDescriptorDto.getType());
        }
    }

    public static DescriptorHelper getInstance(Proxies proxies) {
        if (instance == null) {
            instance = new DescriptorHelper(proxies);
        }
        return instance;        
    }

    public String getRegisteredType(String shortCiType) {
        return isTypeRegistered(shortCiType) ? shortCiType : null;
    }

    public boolean isTypeRegistered(final String shortCiType) {
        return simpleNameMap.contains(shortCiType);
    }

    public static void describe(final String typeName) {
        instance.describeType(typeName);
    }

    private void describeType(final String shortType) {
        final String type = getRegisteredType(shortType);
        if (type == null) {
            out.println();
            out.printf("No ConfigurationItem named \"%s\" found.\n", shortType);
            out.println();
            return;
        }
        ConfigurationItemDescriptorDto ciDescriptor = new ResponseExtractor(descriptorProxy.find(type)).getEntity();

        out.println();
        out.printf("ConfigurationItem %s:\n", ciDescriptor.getType());
        out.printf("Description: %s\n", ciDescriptor.getDescription());
        out.println("Control tasks:");
        for (ConfigurationItemMethodDescriptorDto controlTask : ciDescriptor.getControlTasks()) {
            out.printf("\t. %s: %s\n", controlTask.getName(), controlTask.getDescription());
        }
        out.println("Properties:");
        for (ConfigurationItemPropertyDescriptorDto property : ciDescriptor.getPropertyDescriptors()) {
            out.printf("\t%s %s(%s): %s\n", getPropertyModifier(property), property.getName(), getType(property), property.getDescription());
            if (property.getType() == ENUM) {
                out.printf("\t\tValues: %s\n", Arrays.toString(property.getEnumValues()));
            }
        }
        out.println();
        out.println("Properties marked with a '!' are required for discovery.");
        out.println("Properties marked with a '*' are required.");
        out.println();
    }

	private String getPropertyModifier(ConfigurationItemPropertyDescriptorDto property) {
		if (property.isDiscoveryRequired()) {
			return "!";
		}
		return property.isRequired() ? "*" : "-";
	}

    private String getType(final ConfigurationItemPropertyDescriptorDto property) {
        final ConfigurationItemPropertyKind type = property.getType();
        switch (type) {
            case CI:
                return property.getPropertyClassname();
            case SET_OF_CI:
                return format("Set<%s>", property.getCollectionMemberClassname());
            case LIST_OF_CI:
                return format("List<%s>", property.getCollectionMemberClassname());
            case SET_OF_STRING:
                return "Set<String>";
            case LIST_OF_STRING:
                return "List<String>";
            default:
                return property.getType().name();
        }
    }

	public Set<String> types() {
		return simpleNameMap;
	}
}
