package com.xebialabs.deployit.cli;

import com.google.common.io.Closeables;
import com.xebialabs.deployit.cli.api.Proxies;
import com.xebialabs.deployit.cli.help.HelpScanner;
import com.xebialabs.deployit.cli.ssl.SelfSignedCertificateAcceptingSocketFactory;
import jline.console.ConsoleReader;
import jline.Terminal;
import jline.TerminalFactory;
import nl.javadude.scannit.Configuration;
import nl.javadude.scannit.Scannit;
import nl.javadude.scannit.scanner.TypeAnnotationScanner;

import org.apache.commons.httpclient.HostConfiguration;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.protocol.Protocol;
import org.apache.commons.httpclient.protocol.ProtocolSocketFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.script.ScriptException;
import java.io.*;
import java.lang.reflect.Constructor;
import java.net.MalformedURLException;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

import static com.xebialabs.deployit.cli.CliOptions.EXTENSION_DIR;
import static com.xebialabs.deployit.cli.CliOptions.parse;
import static java.lang.String.format;

public class Cli {
	private static final AtomicReference<Properties> properties = new AtomicReference<Properties>();
	private static ConsoleReader consoleReader;
	private static final FilenameFilter CLI_EXTENSION_FILTER = new FilenameFilter() {
		@Override
		public boolean accept(final File dir, final String name) {
			return name.endsWith(".py") || name.endsWith(".cli");
		}
	};
	private ScriptEngineBuilder scriptEngine;
	private CliOptions options;
    private static Authentication authentication;

    public Cli(CliOptions options) throws Exception {
		this.options = options;
		scriptEngine = createEngine();
		initialize();		
	}

	private void printBanner() {
		bannerPrint("Welcome to the Deployit Jython CLI!");
		bannerPrint("Type 'help' to learn about the objects you can use to interact with Deployit.");
        bannerPrint("");
	}

	private void initialize() throws Exception {
        setAuthentication(createCredentials());
		setupSecureCommunications();
		authentication.init(this.options);
		attemptToConnectToServer(authentication);
		printBanner();
		Proxies proxies = createAndRegisterProxies(authentication);
		registerCliObjects(proxies);
	}

	private void setupSecureCommunications() {
		if (options.isSecured()) {
			Protocol easyhttps = new Protocol("https", (ProtocolSocketFactory) new SelfSignedCertificateAcceptingSocketFactory(), 443);
			Protocol.registerProtocol("https", easyhttps);
		}
	}

	private Proxies createAndRegisterProxies(Authentication client) {
		Proxies proxies = new Proxies(options, client);
		if (options.isExposeProxies()) {
			System.out.println("Exposing Proxies!");
			scriptEngine.put("proxies", proxies);
		}
		return proxies;
	}

	public static void main(String[] args) throws Exception {
		CliOptions options = parse(args);
		if (options == null) return;

		consoleReader = setupConsole();
		new Cli(options).getNewInterpreter().interpret();
	}

	private static ConsoleReader setupConsole() throws IOException {
		final Terminal terminal = TerminalFactory.create();
		Runtime.getRuntime().addShutdownHook(new Thread() {
			@Override
			public void run() {
				try {
	                terminal.restore();
                } catch (Exception ignored) {
                }
			}
		});
		ConsoleReader cr = new ConsoleReader();
		cr.setExpandEvents(false);
		return cr;
	}

	private ScriptEngineBuilder createEngine() {
        return new ScriptEngineBuilder();
	}


	public Interpreter getNewInterpreter() throws ScriptException, FileNotFoundException {
		final Interpreter interpreter = new Interpreter(consoleReader, scriptEngine, options);
		readExtensions(interpreter);
		return interpreter;
	}

	private void registerCliObjects(final Proxies proxies) throws Exception {
		final Scannit scannit = new Scannit(Configuration.config().with(new TypeAnnotationScanner()).scan("com.xebialabs"));
	    final Set<Class<?>> classes = scannit.getTypesAnnotatedWith(CliObject.class);
	    for (Class<?> cliObject : classes) {
	        final Constructor<?> constructor = cliObject.getConstructor(Proxies.class);
	        final Object o = constructor.newInstance(proxies);
	        final String name = cliObject.getAnnotation(CliObject.class).name();
	        scriptEngine.put(name, o);
	    }
		if (!options.isQuiet()) {
	        HelpScanner.printHelp(classes);
		}
	}


	public Authentication createCredentials() throws IOException {
		final Authentication authentication = new Authentication();
		if (options.isUsernameOnCommandline()) {
			authentication.username = options.getUsername();
		} else if (deployitConfigurationFileExists()) {
			authentication.username = readFromProperties("cli.username");
		} else {
			authentication.username = consoleReader.readLine("username >");
		}

		if (options.isPasswordOnCommandline()) {
			authentication.password = options.getPassword();
		} else if (deployitConfigurationFileExists()) {
			authentication.password = readFromProperties("cli.password");
		} else {
			authentication.password = consoleReader.readLine("password >", '\0');
		}

		return authentication;
	}

	private boolean deployitConfigurationFileExists() {
		return options.getConfigurationFile().exists();
	}

	private String readFromProperties(final String key) throws IOException {
		if (properties.get() == null) {
			Properties props = new Properties();
			FileInputStream inStream = new FileInputStream(options.getConfigurationFile());
			try {
				props.load(inStream);
			} finally {
				Closeables.closeQuietly(inStream);
			}
			properties.set(props);
		}

		return properties.get().getProperty(key);
	}

	private void attemptToConnectToServer(Authentication authentication) {
		String urlToConnectTo = options.getUrl();
		System.out.println("Connecting to the Deployit server at " + urlToConnectTo + "...");
		try {
			final int responseCode = authentication.getHttpClient().executeMethod(new GetMethod(urlToConnectTo + "/server/info"));
			if (responseCode == 200) {
				System.out.println("Succesfully connected.");
			} else if (responseCode == 401 || responseCode == 403) {
				throw new IllegalStateException("You were not authenticated correctly, did you use the correct credentials?");
			} else {
				throw new IllegalStateException("Could contact the server at " + urlToConnectTo + " but received an HTTP error code, " + responseCode);
			}
		} catch (MalformedURLException mue) {
			throw new IllegalStateException("Could not contact the server at " + urlToConnectTo, mue);
		} catch (IOException e) {
			throw new IllegalStateException("Could not contact the server at " + urlToConnectTo, e);
		}
	}

	private void readExtensions(Interpreter interpreter) throws ScriptException, FileNotFoundException {
	    final File extensionDir = new File(EXTENSION_DIR);
	    if (!extensionDir.exists() || !extensionDir.isDirectory()) {
	        System.out.println("No extension directory present.");
	        return;
	    }

	    final File[] files = extensionDir.listFiles(CLI_EXTENSION_FILTER);

	    for (File extension : files) {
	        bannerPrint("Reading extension: " + extension);
	        interpreter.evaluate(new FileReader(extension));
	    }
	}

	public static class Authentication {
		String username;
		String password;

        HttpClient httpClient;
        
		void init(final CliOptions options) {
			httpClient = new HttpClient();			
            addCredentials();
			httpClient.getParams().setAuthenticationPreemptive(true);
			httpClient.getParams().setConnectionManagerTimeout(10000); 
			HostConfiguration hostConfiguration = new HostConfiguration();
			hostConfiguration.setHost(options.getHost(), options.getPort(), format("http%s", options.isSecured() ? "s" : ""));
			httpClient.setHostConfiguration(hostConfiguration);
		}

        private void addCredentials() {
            final UsernamePasswordCredentials credentials = new UsernamePasswordCredentials(username, password);
            httpClient.getState().setCredentials(AuthScope.ANY, credentials);
        }

        public void logout() {
            this.username = null;
            this.password = null;
            httpClient.getState().clearCookies();
            httpClient.getState().clearCredentials();
        }

        public String getUserName() {
        	return this.username;
        }

        public void loginAs(String username, String password) {
            if (httpClient.getState().getCredentials(AuthScope.ANY) != null) {
                logger.error("You're still logged in as another user, please logout first.");
                return;
            }

            this.username = username;
            this.password = password;

            addCredentials();
            httpClient.getState().clearCookies();
        }

        public HttpClient getHttpClient() {
            return httpClient;
        }

        private static final Logger logger = LoggerFactory.getLogger(Cli.Authentication.class);
    }

    public static Authentication getAuthentication() {
        return authentication;
    }

    static void setAuthentication(Authentication authentication) {
        Cli.authentication = authentication;
    }
	
	void bannerPrint(String line) {
		if (!options.isQuiet()) {
			System.out.println(line);
		}
	}
}
