package com.xebialabs.deployit.cli.api.internal;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.UnsupportedCharsetException;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.io.Closeables;
import com.xebialabs.deployit.booter.remote.HttpClientHolder;
import com.xebialabs.deployit.booter.remote.resteasy.DeployitClientException;
import com.xebialabs.deployit.engine.api.dto.FileUpload;
import com.xebialabs.deployit.engine.xml.XStreamReaderWriter;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

/**
 * Helper that can both upload packages from the client and import packages on the server.
 */
public class ImportHelper {

    /**
     * Check whether the packageToImport parameter refers to a file by checking for its existence. If it is not a file,
     * it should be a package in the importablePackages directory on the server, and should be importable from there.
     *
     * @param packageToImport
     * @return
     * @throws IOException
     */
    public ConfigurationItem doImport(String baseUri, String packageToImport) {
        try {
            URL url = new URL(packageToImport);
            if (url.getProtocol().equals("file")) {
                return processFilePath(baseUri, url.getFile());
            } else {
                HttpPost httpPost = new HttpPost(baseUri + "/package/fetch");
                httpPost.setEntity(new StringEntity(url.toString(), ContentType.create("application/xml", "UTF-8")));
                return executeAndExtract(httpPost);
            }
        } catch (MalformedURLException e) {
            return processFilePath(baseUri, packageToImport);
        } catch (UnsupportedCharsetException e) {
            throw new RuntimeException(e);
        }
    }

    private ConfigurationItem executeAndExtract(HttpPost httpPost) {
        HttpResponse execute = null;
        try {
            HttpClientHolder instance = HttpClientHolder.getInstance();
            HttpClient httpClient = HttpClientHolder.httpClient();
            execute = httpClient.execute(instance.getHost(), httpPost);
            int i = execute.getStatusLine().getStatusCode();
            if (i < 300) {
                // Ok, it was created.
                return extract(execute);
            } else
                throw new DeployitClientException("Error! (" + i + ")", i);
        } catch (IOException e) {
            throw new RuntimeException(e);
        } finally {
            if (execute != null) {
                try {
                    EntityUtils.consume(execute.getEntity());

                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }

    protected ConfigurationItem extract(HttpResponse postMethod) throws IOException {
        return (ConfigurationItem) XStreamReaderWriter.getConfiguredXStream().fromXML(postMethod.getEntity().getContent());
    }

    private ConfigurationItem processFilePath(String baseUri, String path) {
        final File file = new File(path);
        if (file.exists()) {
            final FileUpload fileUpload = new FileUpload();
            FileInputStream fileData = null;
            try {
                fileData = new FileInputStream(file);
                fileUpload.setFileData(fileData);
                HttpPost httpPost = new HttpPost(baseUri + "/package/upload/" + file.getName());
                MultipartEntity entity = new MultipartEntity();
                entity.addPart("fileData", new FileBody(file));
                httpPost.setEntity(entity);
                return executeAndExtract(httpPost);
            } catch (IOException e) {
                logger.error("The file \'{}\' exists, but could not be read for upload; perhaps it's not an archive.", file.toString());
                return null;
            } finally {
                Closeables.closeQuietly(fileData);
            }
        } else {
            if (!file.isAbsolute()) {
                HttpPost postMethod = new HttpPost(baseUri + "/package/import/" + path);
                return executeAndExtract(postMethod);
            } else {
                logger.error("The file \'{}\' does not exist at the given absolute path", file.toString());
                return null;
            }
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(ImportHelper.class);
}
