package com.xebialabs.deployit.cli.api;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import org.jboss.resteasy.client.ClientResponseFailure;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.core.api.dto.RolePrincipals;
import com.xebialabs.deployit.engine.api.dto.ServerInfo;
import com.xebialabs.deployit.engine.api.security.Permission;
import com.xebialabs.deployit.engine.api.security.User;

import static com.google.common.collect.Lists.newArrayList;

@CliObject(name = "security")
@ClassHelp(description = "Access to the security settings of XL Deploy.")
public class SecurityClient extends DocumentedObject {

    private ProxiesInstance proxies;
    private DeployitCommunicator communicator;

    public SecurityClient() {}

    public SecurityClient(ProxiesInstance proxies) {
        communicator = proxies.getCommunicator();
        this.proxies = proxies;
    }

    @MethodHelp(description = "Logout the currently logged in user, can only perform further actions after a login")
    public void logout() {
        try {
            proxies.getServer().logout();
        } catch (ClientResponseFailure e) {
            int status = e.getResponse().getStatus();
            // Response 401: Assume we are logged out
            // Response 403: Credentials are wrong, just invalidating them below
            if (status != 401 && status != 403) {
                throw e;
            }
        }
        communicator.getHttpClientHolder().logout();
    }

    @MethodHelp(description = "Login a user", parameters = {
        @ParameterHelp(name = "username", description = "The username"),
        @ParameterHelp(name = "password", description = "The password")
    })
    public void login(String username, String password) {
        logger.info("Logging in as {}", username);
        communicator.getHttpClientHolder().loginAs(username, password);
        try {
            @SuppressWarnings("unused")
            ServerInfo info = proxies.getServer().getInfo();
        } catch (Exception e) {
            communicator.getHttpClientHolder().logout();
            throw new IllegalStateException("You're not authorized with these credentials. (" + username + ")", e);
        }
    }

    @MethodHelp(description = "Create a user with the specified name and password", parameters = {
        @ParameterHelp(name = "username", description = "The username"),
        @ParameterHelp(name = "password", description = "The password")
    }, returns = "The created user")
    public User createUser(String username, String password) {
        return createUser(username, password, false);
    }

    public User createUser(final String username, final String password, final boolean admin) {
        User user = new User(username, admin);
        user.setPassword(password);
        return proxies.getUser().create(username, user);
    }

    @MethodHelp(description = "Read a user so that he/she can be modified.", parameters = {
        @ParameterHelp(name = "username", description = "The username of the user to read")
    }, returns = "The read user")
    public User readUser(String username) {
        return proxies.getUser().read(username);
    }

    @MethodHelp(description = "Modify the (password of) the user.", parameters = {
        @ParameterHelp(name = "user", description = "The updated user object.")
    })
    public void modifyUser(User user) {
        proxies.getUser().modifyPassword(user.getUsername(), user);

    }

    @MethodHelp(description = "Delete a user.", parameters = {
        @ParameterHelp(name = "username", description = "the username of the user to be deleted")
    })
    public void deleteUser(String username) {
        proxies.getUser().delete(username);
    }

    @MethodHelp(description = "Grant a permission to a role", parameters = {
        @ParameterHelp(name = "permission", description = "The permission to grant"),
        @ParameterHelp(name = "roleName", description = "The role")
    })
    public void grant(String permission, String roleName) {
        doGrant(permission, roleName, "global");
    }

    @MethodHelp(description = "Grant a permission to a role on a group of configuration items", parameters = {
        @ParameterHelp(name = "permission", description = "The permission to grant"),
        @ParameterHelp(name = "roleName", description = "The role"),
        @ParameterHelp(name = "configurationItems", description = "A list of configuration items to which the permission should apply")
    })
    public void grant(String permission, String roleName, List<String> configurationItems) {
        List<String> ids = removeTrailingSlashes(configurationItems);
        for (String id : ids) {
            doGrant(permission, roleName, id);
        }
    }

    @MethodHelp(description = "Revoke a permission from a role", parameters = {
        @ParameterHelp(name = "permission", description = "The permission to revoke"),
        @ParameterHelp(name = "roleName", description = "The role")
    })
    public void revoke(String permission, String roleName) {
        doRevoke(permission, roleName, "global");
    }

    @MethodHelp(description = "Revoke a permission from a role on a group of configuration items", parameters = {
        @ParameterHelp(name = "permission", description = "The permission to grant"),
        @ParameterHelp(name = "roleName", description = "The role"),
        @ParameterHelp(name = "configurationItems", description = "A list of configuration items from which the permission should be removed")
    })
    public void revoke(String permission, String roleName, List<String> configurationItems) {
        List<String> ids = removeTrailingSlashes(configurationItems);
        for (String id : ids) {
            doRevoke(permission, roleName, id);
        }
    }

    private void doGrant(String permission, String role, String id) {
        proxies.getPermissions().grant(permission, id, role);
    }

    private void doRevoke(String permission, String role, String id) {
        proxies.getPermissions().revoke(permission, id, role);
    }

    @MethodHelp(description = "Check whether a permission is granted to the logged in user on an id.", parameters = {
        @ParameterHelp(name = "permission", description = "The permission to check"),
        @ParameterHelp(name = "id", description = "The path of the CI to check the permission on.") },
        returns = ("true if the logged in user is granted the permission on the node."))
    public boolean hasPermission(String permission, String id) {
        return proxies.getPermissions().isGrantedToMe(permission, id);
    }

    @MethodHelp(description = "Check whether a permission is granted to a role on an id.", parameters = {
        @ParameterHelp(name = "role", description = "The role to check for"),
        @ParameterHelp(name = "permission", description = "The permission to check"),
        @ParameterHelp(name = "id", description = "The path of the CI to check the permission on.") },
        returns = ("true if the role has the permission on the node."))
    public boolean isGranted(String role, String permission, String id) {
        return proxies.getPermissions().isGranted(permission, id, role);
    }

    @MethodHelp(description = "Check whether a global permission is granted to a role.", parameters = {
        @ParameterHelp(name = "role", description = "The role to check for"),
        @ParameterHelp(name = "permission", description = "The permission to check") },
        returns = ("true if the role has the global permission granted."))
    public boolean isGranted(String role, String permission) {
        return proxies.getPermissions().isGranted(permission, "global", role);
    }

    @MethodHelp(description = "Get permissions granted to a role", parameters = {
        @ParameterHelp(name = "role", description = "The role whose permissions are to be retrieved") },
        returns = ("A Map containing the nodes that the role has permissions on with the associated granted permissions"))
    public Map<String, Collection<String>> getPermissions(final String role) {
        return proxies.getPermissions().getGrantedPermissions(role);
    }

    @MethodHelp(description = "Get permissions granted to the currently logged in user", parameters = {},
            returns = ("A Map containing the nodes that the logged in user has permissions on with the associated granted permissions"))
    public Map<String, Collection<String>> getPermissions() {
        return proxies.getPermissions().getMyGrantedPermissions();
    }

    @MethodHelp(description = "List all the permissions available in XL Deploy")
    public void printPermissions() {
        System.out.println("Available permissions are:");
        for (Permission permission : proxies.getReferenceData().listPermissions()) {
            System.out.println(permission.getPermissionName());
        }
    }

    @MethodHelp(description = "Create or update a role with assigned principals.", parameters = {
        @ParameterHelp(name = "roleName", description = "The role name"),
        @ParameterHelp(name = "principals", description = "The assigned principals")
    })
    public void assignRole(final String roleName, List<String> principals) {
        if (principals == null || principals.isEmpty()) {
            proxies.getRoleService().create(roleName);
        } else {
            for (String principal : principals) {
                proxies.getRoleService().assign(roleName, principal);
            }
        }
    }

    @MethodHelp(description = "Remove a role.", parameters = {
        @ParameterHelp(name = "roleName", description = "The role name")
    })
    public void removeRole(String roleName) {
        proxies.getRoleService().delete(roleName);
    }

    @MethodHelp(description = "Get all existing roles in XL Deploy.", returns = "The list of role names")
    public List<String> getRoleNames() {
        return proxies.getRoleService().list();
    }

    public void renameRole(String oldName, String newName) {
        proxies.getRoleService().rename(oldName, newName);
    }

    @MethodHelp(description = "Get the principals assigned to a specific role.", parameters = {
        @ParameterHelp(name = "roleName", description = "The role name")
    }, returns = "A list of principals assigned to the role")
    public List<String> getRoleAssignments(String roleName) {
        List<RolePrincipals> assignments = proxies.getInternalSecurity().readRolePrincipals();
        for (RolePrincipals assignment : assignments) {
            if (assignment.getRole().getName().equals(roleName)) {
                return assignment.getPrincipals();
            }
        }
        return newArrayList();
    }

    private static List<String> removeTrailingSlashes(final List<String> repositoryEntityIds) {
        List<String> cleanRepositoryEntityIds = newArrayList();
        for (String repositoryEntityId : repositoryEntityIds) {
            cleanRepositoryEntityIds.add(repositoryEntityId.replaceFirst("/*$", ""));
        }
        return cleanRepositoryEntityIds;
    }

    private static final Logger logger = LoggerFactory.getLogger(SecurityClient.class);
}
