package com.xebialabs.deployit.cli;

import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.cli.help.HelpScanner;
import com.xebialabs.deployit.cli.util.Closeables;
import jline.console.ConsoleReader;
import org.python.core.Py;
import org.python.core.PyBaseExceptionDerived;
import org.python.core.PyException;

import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.ScriptException;
import javax.script.SimpleScriptContext;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.Reader;
import java.util.Set;

import static java.lang.System.out;

public class Interpreter {
    private final Set<String> cliObjectNames;
    private ScriptEngine scriptEngine;
    private CliOptions options;
    private ScriptContext scriptContext;
    private ConsoleReader consoleReader;
    private DeployitCommunicator communicator;

    public Interpreter(ConsoleReader consoleReader, ScriptEngineBuilder scriptEngineBuilder, DeployitCommunicator communicator, CliOptions options, Set<String> cliObjectNames) {
        this.consoleReader = consoleReader;
        this.communicator = communicator;
        this.scriptEngine = scriptEngineBuilder.build(options);
        this.options = options;
        this.scriptContext = new SimpleScriptContext();
        ScriptContext context = scriptEngine.getContext();
        this.scriptContext.setBindings(context.getBindings(ScriptContext.ENGINE_SCOPE), ScriptContext.GLOBAL_SCOPE);
        this.cliObjectNames = cliObjectNames;
    }

    void interpret() throws Exception {
        try {
            if (options.interpretSourceFile()) {
                evaluateFile(options.getSourceFileName());
            } else {
                try {
                    advancedRepl(consoleReader);
                } catch (Exception e) {
                    throw new IllegalStateException("The REPL gave an exception!", e);
                }
            }
        } catch (FileNotFoundException e) {
            System.err.println("File " + options.getSourceFileName() + " could not be read, reason:" + e.getMessage());
            System.exit(1);
        } catch (ScriptException e) {
            int exitCode;
            if (sysExitInvoked(e)) {
                exitCode = getSysExitCode(e);
            } else {
                printScriptException(e);
                exitCode = 1;
            }
            System.exit(exitCode);
        }
    }

    public void evaluateFile(String fileName) throws FileNotFoundException, ScriptException {
        scriptContext.setAttribute(ScriptEngine.FILENAME, fileName, ScriptContext.ENGINE_SCOPE);
        FileReader reader = new FileReader(fileName);
        try {
            evaluateBuiltins();
            evaluate(reader);
        } finally {
            Closeables.closeQuietly(reader);
        }
    }

    private boolean sysExitInvoked(ScriptException e) {
        return e.getCause() instanceof PyException && ((PyException) e.getCause()).type.equals(Py.SystemExit);
    }

    private int getSysExitCode(ScriptException e) {
        return ((PyBaseExceptionDerived) ((PyException) e.getCause()).value).getMessage().asInt();
    }

    public Object evaluate(Reader reader) throws ScriptException {
        return scriptEngine.eval(reader, scriptContext);
    }

    Object evaluate(String expression) throws ScriptException {
        if (expression == null) {
            return null;
        }
        return scriptEngine.eval(expression, scriptContext);
    }

    private void advancedRepl(ConsoleReader consoleReader) throws Exception {
        evaluateBuiltins();
        for (; ; ) {
            final String line = consoleReader.readLine(getPrompt());
            final String trimmed = line != null ? line.trim() : null;
            if (trimmed == null || trimmed.startsWith("quit") || trimmed.startsWith("exit")) {
                exit();
                break;
            } else if (trimmed.startsWith("help")) {
                HelpScanner.printHelp();
                continue;
            }
            try {
                Object evaluationResult = evaluate(line);
                if (evaluationResult instanceof String) {
                    out.println((String) evaluationResult);
                } else if (evaluationResult != null) {
                    out.println(evaluationResult.getClass().getSimpleName() + ": " + evaluationResult);
                } else {
                    //FIXME: not sure what to give back; maybe out.println("Ok"); ?
                }
            } catch (ScriptException e) {
                printScriptException(e);
            }
        }
    }

    private void evaluateBuiltins() {
        try {
            evaluate("import __builtin__");
            cliObjectNames.forEach(cliObject -> {
                try {
                    evaluate(String.format("__builtin__.%s = %s", cliObject, cliObject));
                } catch (ScriptException e) {
                    printScriptException(e);
                }
            });
        } catch (ScriptException e) {
            printScriptException(e);
        }

    }

    private void printScriptException(ScriptException e) {
        Throwable cause = e.getCause();
        if (cause == null) {
            System.err.println(e);
        } else if (cause instanceof PyException) {
            if (cause.getCause() == null) {
                System.err.println(cause);
            } else {
                System.err.println(cause.getCause());
            }
        } else {
            System.err.println(cause);
        }
    }

    protected void exit() throws Exception {
        System.out.println("Bye bye!");
    }

    String getPrompt() {
        String prompt = "deployit > ";
        String username = communicator.getHttpClientHolder().getUserName();
        if (username != null) {
            prompt = username + " > ";
        }
        return prompt;
    }

    public ScriptContext getScriptContext() {
        return scriptContext;
    }
}
