package com.xebialabs.deployit.rest.test.api {

import com.xebialabs.deployit.service.importer.source.FileSource
import java.io.File
import com.xebialabs.deployit.plugin.test.v3.{DummyEar, DummyHost, DummyJeeServer, DummyEarWithAllProperties, ConfigurationFiles}
import com.xebialabs.deployit.repository._
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.security.permission.Permission
import scala.collection.JavaConversions._
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary
import collection.mutable.Set
import com.xebialabs.deployit.deployment.planner.DeltaSpecificationBuilder
import com.xebialabs.deployit.service.deployment.{DeploymentOperationCalculator, DeployedGenerator}
import com.xebialabs.deployit.test.deployment.DeployitTester
import com.xebialabs.deployit.task.deployment.InitialDeploymentTask
import java.util.concurrent.CountDownLatch
import com.xebialabs.deployit.plugin.api.deployment.execution.Plan
import com.xebialabs.deployit.task.TaskForReporting
import com.xebialabs.deployit.plugin.TestServiceHolder
import scala.collection.mutable.HashSet

trait Generator {

	val tester = DeployitTester.build()

	def user(name: String) {
		TestServiceHolder.getUserService.create(name, name)
	}

	def create(c: Creator => Unit) {
		val changeSet: ChangeSet = new ChangeSet
		val creator = new Creator(changeSet)
		c.apply(creator)
		RepositoryServiceHolder.getRepositoryService.execute(changeSet)
		creator.cleanUp()
	}

	private def read[T](id: String, workDir: WorkDir): T = {
		RepositoryServiceHolder.getRepositoryService.read(id, workDir)
	}

	def grant(user: String)(perm: String, onCi: String = "") {
		Permission.find(perm).getPermissionHandler.grant(user, if (onCi == "") List.empty[String] else List(onCi))
	}

	def task(c: TaskForReporting => Unit) {
		val taskForReporting: TaskForReporting = new TaskForReporting()
		c.apply(taskForReporting)
		TestServiceHolder.getTaskArchive.archiveTask(taskForReporting)
	}

	def deploy(dpId: String, envId: String) = {
		val workDir = TestServiceHolder.getWorkDirFactory.newWorkDir()
		val dp: DeploymentPackage = read(dpId, workDir)
		val env: Environment = read(envId, workDir)
		val set: Set[Deployed[_ <: Deployable, _ <: Container]] = generateDeployeds(dp, env)
		val specification = createSpecification(dp, env, set)

		val plan: Plan = tester.resolvePlan(specification)

		val latch = new CountDownLatch(1)

		val initialDeploymentTask: InitialDeploymentTask = new InitialDeploymentTask(specification, plan.getSteps, RepositoryServiceHolder.getRepositoryService, workDir) {
			override def doAfterTaskStateChangedToDone() {
				try {
					super.doAfterTaskStateChangedToDone()
				} finally {
					latch.countDown()
				}
			}

			override def doAfterTaskStateChangedToAborted() {
				try {
					super.doAfterTaskStateChangedToAborted()
				} finally {
					latch.countDown()
				}
			}
		}

		val taskId: String = TestServiceHolder.getExecutionEngine.register(initialDeploymentTask)
		TestServiceHolder.getExecutionEngine.execute(taskId)
		latch.await()

		taskId
	}

	private def generateDeployeds(dp: DeploymentPackage, env: Environment) = {
		val deployedGenerator: DeployedGenerator = new DeployedGenerator(RepositoryServiceHolder.getRepositoryService)
		val dict: ConsolidatedDictionary = ConsolidatedDictionary.create(env.getDictionaries)
		val set: Set[Deployed[_ <: Deployable, _ <: Container]] = new HashSet[Deployed[_ <: Deployable, _ <: Container]]
		for (d <- dp.getDeployables; c <- env.getMembers) {
			val deployed = deployedGenerator.generateMostSpecificDeployed(d, c, dict)
			if (deployed != null)
				set += deployed.asInstanceOf[Deployed[_ <: Deployable, _ <: Container]]
		}
		set
	}

	private def createSpecification(dp: DeploymentPackage, env: Environment, set: Set[Deployed[_ <: Deployable, _ <: Container]]) = {
		val builder: DeltaSpecificationBuilder = DeltaSpecificationBuilder.newSpecification.initial(dp, env)
		DeploymentOperationCalculator.calculate(builder, Set.empty[Deployed[_ <: Deployable, _ <: Container]], set)
		builder.build()
	}

	def importPackage(id: String) {
		val dir: File = TestServiceHolder.getImporterService.getImportablePackageDirectory
		val packageId: String = TestServiceHolder.getImporterService.importPackage(new FileSource(new File(dir, id).getPath, false))
	}

	class Creator(val changeSet: ChangeSet) {
		def ci[A <: ConfigurationItem](id: String, t: A)(c: A => Unit): A = {
			t.setId(id)
			c.apply(t)
			changeSet.getCreateCis.add(t)
			t
		}

		def environment(id: String)(members: List[_ <: Container]): Environment = {
			env(id) {
				e =>
					members.foreach(e.addMember(_))
			}
		}

		def deploymentPackage(id: String, app: Application)(deployables: DeploymentPackage => List[_ <: Deployable]): DeploymentPackage = {
			val dp: DeploymentPackage = new DeploymentPackage()
			dp.setId(app.getId + "/" + id)
			changeSet.getCreateCis.add(dp)
			deployables.apply(dp).foreach(dp.addDeployable(_))
			dp
		}

		def env(id: String): (Environment => Unit) => Environment = {
			ci("Environments/" + id, new Environment()) _
		}

		def server(id: String): (DummyJeeServer => Unit) => DummyJeeServer = {
			ci("Infrastructure/" + id, new DummyJeeServer()) _
		}

		def host(id: String): (DummyHost => Unit) => DummyHost = {
			ci("Infrastructure/" + id, new DummyHost()) _
		}

		def application(id: String): Application = {
			ci("Applications/" + id, new Application) {
				_ =>
					Unit
			}
		}

		def ear(id: String, dp: DeploymentPackage): (DummyEar => Unit) => DummyEar = {
			ci(dp.getId + "/" + id, new DummyEar) _
		}

		def earWithProperties(id: String, dp: DeploymentPackage): (DummyEarWithAllProperties => Unit) => DummyEarWithAllProperties = {
			ci(dp.getId + "/" + id, new DummyEarWithAllProperties) _
		}

		def configurationFile(id: String, dp: DeploymentPackage): (ConfigurationFiles => Unit) => ConfigurationFiles = {
			ci(dp.getId + "/" + id, new ConfigurationFiles) _
		}

		def cleanUp() {
		}
	}

}

}

package com.xebialabs.deployit.task {

import com.xebialabs.deployit.plugin.api.deployment.execution.{DeploymentExecutionContext, DeploymentStep}
import com.xebialabs.deployit.task.DeploymentTask.DeploymentType
import com.xebialabs.deployit.task.Task.State
import com.xebialabs.deployit.task.TaskStepInfo.StepState
import java.util.Calendar
import com.xebialabs.deployit.plugin.api.execution.{ExecutionContext, Step}
import scala.collection.JavaConversions._

class SampleStep extends DeploymentStep {
	def getOrder = 0

	def getDescription = ""

	def execute(ctx: DeploymentExecutionContext) = Step.Result.Success
}

class TestTaskStep(step: Step[_ <: ExecutionContext], state: TaskStepInfo.StepState) extends TaskStep(step, state) {
	override def setState(state: StepState) {
		super.setState(state)
	}

	override def setStartDate(cal: Calendar) {
		super.setStartDate(cal)
	}

	override def setCompletionDate(cal: Calendar) {
		super.setCompletionDate(cal)
	}

	override def setFailureCount(failureCount: Int) {
		super.setFailureCount(failureCount)
	}
}

class TaskForReporting(list: List[TaskStep], state: State) extends DeploymentTask(list, state) {
	def this() = this (List[TaskStep](new TestTaskStep(new SampleStep, TaskStepInfo.StepState.DONE)), State.DONE)

	def getDeploymentType = DeploymentType.INITIAL

	def failed = {
		init(State.CANCELLED, 1, TaskStepInfo.StepState.FAILED)
		this
	}

	def success = {
		init(State.DONE, 0, TaskStepInfo.StepState.DONE)
		this
	}

	def successWithFails = {
		init(State.DONE, 1, TaskStepInfo.StepState.DONE)
		this
	}

	private def init(state: State, fails: Int, stepState: TaskStepInfo.StepState) {
		setState(state)
		setOwner("admin")
		val step: TestTaskStep = getSteps.get(0).asInstanceOf[TestTaskStep]
		step.setFailureCount(fails)
		step.setState(stepState)
	}

	def startedAt(startDate: Calendar) = {
		setStartDate(startDate)
		this
	}

	def completedAt(completionDate: Calendar) = {
		setCompletionDate(completionDate)
		this
	}

	def forEnv(env: String) = {
		setEnvironment(env)
		this
	}

	def forApp(app: String) = {
		setApplicationName(app)
		this
	}

	def forVersion(version: String) = {
		setApplicationVersion(version)
		this
	}

	def withId(id: String) = {
		setId(id)
		this
	}
}

}