package com.xebialabs.deployit.rest.test.api {

import collection.mutable.Set
import scala.collection.JavaConversions._
import scala.collection.mutable.HashSet

import java.io.File
import java.lang.String
import java.util.concurrent.CountDownLatch
import java.util.{List=>JavaList}

import com.google.common.collect.Multimap

import com.xebialabs.deployit.deployment.planner.DeltaSpecificationBuilder
import com.xebialabs.deployit.plugin.TestServiceHolder
import com.xebialabs.deployit.plugin.api.udm._
import com.xebialabs.deployit.plugin.test.v3.{DummyEar, DummyHost, DummyJeeServer, DummyEarWithAllProperties, ConfigurationFiles}
import com.xebialabs.deployit.repository._
import com.xebialabs.deployit.repository.core.Directory
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.{PermissionEditor, Role, RoleService}
import com.xebialabs.deployit.service.deployment.{TypeCalculator, DeploymentOperationCalculator, DeployedGenerator}
import com.xebialabs.deployit.service.importer.source.FileSource
import com.xebialabs.deployit.service.replacement.ConsolidatedDictionary
import com.xebialabs.deployit.test.deployment.{DeltaSpecifications, DeployitTester}
import java.util
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.engine.tasker.TaskSpecification
import com.xebialabs.deployit.engine.spi.execution.{TaskExecutionStateEvent, StepExecutionStateEvent, ExecutionStateListener}
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState
import com.xebialabs.deployit.task.archive.ArchivedTask
import org.joda.time.DateTime
import com.xebialabs.deployit.task.TaskType
import com.xebialabs.deployit.service.deployment.RepositoryUpdateTrigger
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry

trait Generator {

    val tester = DeployitTester.build()

    def user(name: String) {
        TestServiceHolder.getUserService.create(name, name)
    }
    
    def role(name: String, principals: List[String]) {
        val service: RoleService = TestServiceHolder.getRoleService
        val assignments: JavaList[Role] = service.readRoleAssignments()
        val role = new Role(name)
        role.getPrincipalsAssigned.addAll(principals)
        assignments.add(role)
        service.writeRoleAssignments(assignments)
    }

    def create(c: Creator => Unit) {
        val changeSet: ChangeSet = new ChangeSet
        val creator = new Creator(changeSet)
        c.apply(creator)
        RepositoryServiceHolder.getRepositoryService.execute(changeSet)
        creator.cleanUp()
    }

    private def read[T](id: String, workDir: WorkDir): T = {
        RepositoryServiceHolder.getRepositoryService.read(id, workDir)
    }

    def grant(role: String)(perm: String, onCi: String = "") {
        val editor: PermissionEditor = TestServiceHolder.getPermissionEditor
        val permissions: Multimap[Role, Permission] = editor.readPermissions(onCi)
        permissions.put(TestServiceHolder.getRoleService.getRoleForRoleName(role), Permission.find(perm))
        editor.editPermissions(onCi, permissions)
    }

    def task(c: ArchivedTask => Unit) {
        val taskForReporting: ArchivedTask = new ArchivedTask()
        taskForReporting.getMetadata.put("taskType", TaskType.INITIAL.name())
        taskForReporting.setOwner("admin")
        c.apply(taskForReporting)
        TestServiceHolder.getTaskArchive.archive(taskForReporting)
    }

    def deploy(dpId: String, envId: String) = {
        val workDir = TestServiceHolder.getWorkDirFactory.newWorkDir()
        val dp: DeploymentPackage = read(dpId, workDir)
        val env: Environment = read(envId, workDir)
        val set: Set[Deployed[_ <: Deployable, _ <: Container]] = generateDeployeds(dp, env)
        val specification = createSpecification(dp, env, set)

        val plan: util.List[Step] = tester.resolvePlan(specification)

        val latch = new CountDownLatch(1)

        val specification1 = new TaskSpecification("deployment", "admin", plan)
        specification1.getMetadata.put("environment", env.getName)
        specification1.getMetadata.put("environment_id", env.getId)
        specification1.getMetadata.put("application", dp.getApplication.getName)
        specification1.getMetadata.put("version", dp.getName)
        specification1.getMetadata.put("taskType", TaskType.INITIAL.name())
        specification1.getListeners.add(new ExecutionStateListener {
          def taskStateChanged(event: TaskExecutionStateEvent) {
            if (event.currentState() == TaskExecutionState.EXECUTED || event.currentState() == TaskExecutionState.STOPPED) {
              latch.countDown()
            }
          }

          def stepStateChanged(event: StepExecutionStateEvent) {}
        })
        specification1.getListeners.add(new RepositoryUpdateTrigger(specification))
    
        val taskId = TestServiceHolder.getExecutionEngine.register(specification1)

        TestServiceHolder.getExecutionEngine.execute(taskId)
        latch.await()
        if (TestServiceHolder.getExecutionEngine.retrieve(taskId).getState == TaskExecutionState.EXECUTED) {
          TestServiceHolder.getExecutionEngine.archive(taskId)
        }

        taskId
    }

    private def generateDeployeds(dp: DeploymentPackage, env: Environment) = {
        val deployedGenerator: DeployedGenerator = new DeployedGenerator(RepositoryServiceHolder.getRepositoryService, new TypeCalculator())
        val dict: ConsolidatedDictionary = ConsolidatedDictionary.create(env.getDictionaries)
        val set: Set[Deployed[_ <: Deployable, _ <: Container]] = new HashSet[Deployed[_ <: Deployable, _ <: Container]]
        for (d <- dp.getDeployables; c <- env.getMembers) {
            val deployed = deployedGenerator.generateMostSpecificDeployed(d, c, dict)
            if (deployed != null)
                set += deployed.asInstanceOf[Deployed[_ <: Deployable, _ <: Container]]
        }
        set
    }

    private def createSpecification(dp: DeploymentPackage, env: Environment, set: Set[Deployed[_ <: Deployable, _ <: Container]]) = {
        val deployedApp = DeltaSpecifications.createDeployedApplication(dp, env)
        val builder: DeltaSpecificationBuilder = DeltaSpecificationBuilder.newSpecification.initial(deployedApp)
        DeploymentOperationCalculator.calculate(builder, Set.empty[Deployed[_ <: Deployable, _ <: Container]], set)
        builder.build()
    }

    def importPackage(id: String) {
        val dir: File = TestServiceHolder.getImporterService.getImportablePackageDirectory
        val packageId: String = TestServiceHolder.getImporterService.importPackage(new FileSource(new File(dir, id).getPath, false))
    }

    class Creator(val changeSet: ChangeSet) {
        def ci[A <: ConfigurationItem](id: String, t: A)(c: A => Unit): A = {
            t.setId(id)
            c.apply(t)
            changeSet.getCreateCis.add(t)
            t
        }

        def pipeline(id: String): ConfigurationItem = {
            val p: ConfigurationItem = DescriptorRegistry.getDescriptor("release.DeploymentPipeline").newInstance()
            p.setId("Configuration/" + id)
            changeSet.getCreateCis.add(p)
            p            
        }
        
        def environment(id: String)(members: List[_ <: Container]): Environment = {
            env(id) {
                e =>
                    members.foreach(e.addMember(_))
            }
        }

        def deploymentPackage(id: String, app: Application)(deployables: DeploymentPackage => List[_ <: Deployable]): DeploymentPackage = {
            val dp: DeploymentPackage = new DeploymentPackage()
            dp.setId(app.getId + "/" + id)
            changeSet.getCreateCis.add(dp)
            deployables.apply(dp).foreach(dp.addDeployable(_))
            dp
        }

        def env(id: String): (Environment => Unit) => Environment = {
            ci("Environments/" + id, new Environment()) _
        }

        def server(id: String): (DummyJeeServer => Unit) => DummyJeeServer = {
            ci("Infrastructure/" + id, new DummyJeeServer()) _
        }

        def host(id: String): (DummyHost => Unit) => DummyHost = {
            ci("Infrastructure/" + id, new DummyHost()) _
        }

        def application(id: String): Application = {
            ci("Applications/" + id, new Application) {
                _ =>
                    Unit
            }
        }
        
        def directory(fullId: String): Directory = {
            ci(fullId, new Directory)  {
                _ =>
                    Unit
            }
        }

        def dictionary(name: String, entries: Map[String, String]): Dictionary = {
          ci("Environments/" + name, new Dictionary) {
            dictionary => dictionary.setEntries(entries)
          }
        }

        def ear(id: String, dp: DeploymentPackage): (DummyEar => Unit) => DummyEar = {
            ci(dp.getId + "/" + id, new DummyEar) _
        }

        def earWithProperties(id: String, dp: DeploymentPackage): (DummyEarWithAllProperties => Unit) => DummyEarWithAllProperties = {
            ci(dp.getId + "/" + id, new DummyEarWithAllProperties) _
        }

        def configurationFile(id: String, dp: DeploymentPackage): (ConfigurationFiles => Unit) => ConfigurationFiles = {
            ci(dp.getId + "/" + id, new ConfigurationFiles) _
        }

        def cleanUp() {
        }
    }

  implicit def task2dsl(task: ArchivedTask): TaskDsl = new TaskDsl(task)

  class TaskDsl(task: ArchivedTask) {
    def withId(id: String) = {
      task.setId(id)
      task
    }

    def completedAt(date: DateTime) = {
      task.setCompletionDate(date)
      task
    }

    def startedAt(date: DateTime) = {
      task.setStartDate(date)
      task
    }

    def forEnv(env: String) = {
      task.getMetadata.put("environment", env)
      task.getMetadata.put("environment_id", env)
      task
    }

    def forApp(app: String) = {
      task.getMetadata.put("application", app)
      task
    }

    def forVersion(version: String) = {
      task.getMetadata.put("version", version)
      task
    }

    def forDeploymentType(taskType: TaskType) = {
      task.getMetadata.put("taskType", taskType.toString)
      task
    }

    def success = {
      task.setState(TaskExecutionState.DONE)
      task
    }

    def successWithFails = {
      task.setState(TaskExecutionState.DONE)
      task.setFailureCount(1)
      task
    }

    def failed = {
      task.setState(TaskExecutionState.CANCELLED)
      task.setFailureCount(1)
      task
    }
  }
}
}