package com.xebialabs.deployit.cli;

import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.booter.remote.resteasy.DeployitClientException;
import com.xebialabs.deployit.cli.help.HelpScanner;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import jline.console.ConsoleReader;
import org.python.core.Py;
import org.python.core.PyBaseExceptionDerived;
import org.python.core.PyException;

import javax.script.ScriptContext;
import javax.script.ScriptEngine;
import javax.script.ScriptException;
import javax.script.SimpleScriptContext;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Set;

import static java.lang.System.out;

public class Interpreter {
    private final Set<String> cliObjectNames;
    private final ScriptEngine scriptEngine;
    private final CliOptions options;
    private final ScriptContext scriptContext;
    private final ConsoleReader consoleReader;
    private final DeployitCommunicator communicator;

    public Interpreter(ConsoleReader consoleReader, ScriptEngineBuilder scriptEngineBuilder, DeployitCommunicator communicator, CliOptions options, Set<String> cliObjectNames) {
        this.consoleReader = consoleReader;
        this.communicator = communicator;
        this.scriptEngine = scriptEngineBuilder.build();
        this.options = options;
        this.scriptContext = new SimpleScriptContext();
        ScriptContext context = scriptEngine.getContext();
        this.scriptContext.setWriter(new UnbufferedPrintWriter(System.out));
        this.scriptContext.setErrorWriter(new UnbufferedPrintWriter(System.err));
        this.scriptContext.setBindings(context.getBindings(ScriptContext.ENGINE_SCOPE), ScriptContext.GLOBAL_SCOPE);
        this.cliObjectNames = cliObjectNames;
    }

    void interpret() {
        String currentlyEvaluatedFile = null;
        try {
            if (!options.getScriptArgs().isEmpty()) {
                String[] value = options.getScriptArgs().toArray(new String[0]);
                scriptContext.setAttribute(ScriptEngine.ARGV, value, ScriptContext.ENGINE_SCOPE);
            }
            if (options.interpretSourceFile()) {
                for (String scriptFileName : options.getSourceFileNames()) {
                    currentlyEvaluatedFile = scriptFileName;
                    evaluateFile(scriptFileName);
                }
            } else {
                try {
                    advancedRepl(consoleReader);
                } catch (Exception e) {
                    throw new IllegalStateException("The REPL gave an exception!", e);
                }
            }
        } catch (FileNotFoundException e) {
            System.err.println("File " + currentlyEvaluatedFile + " could not be read, reason:" + e.getMessage());
            System.exit(1);
        } catch (ScriptException e) {
            int exitCode;
            if (sysExitInvoked(e)) {
                exitCode = getSysExitCode(e);
            } else {
                printScriptException(e);
                exitCode = 1;
            }
            System.exit(exitCode);
        }
    }

    public void evaluateFile(String fileName) throws FileNotFoundException, ScriptException {
        scriptContext.setAttribute(ScriptEngine.FILENAME, fileName, ScriptContext.ENGINE_SCOPE);
        try (FileReader reader = new FileReader(fileName, StandardCharsets.UTF_8)) {
            evaluate(reader);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private boolean sysExitInvoked(ScriptException e) {
        return e.getCause() instanceof PyException && ((PyException) e.getCause()).type.equals(Py.SystemExit);
    }

    private int getSysExitCode(ScriptException e) {
        return ((PyBaseExceptionDerived) ((PyException) e.getCause()).value).getMessage().asInt();
    }

    public Object evaluate(Reader reader) throws ScriptException {
        evaluateBuiltins();
        return scriptEngine.eval(reader, scriptContext);
    }

    Object evaluate(String expression) throws ScriptException {
        if (expression == null) {
            return null;
        }
        return scriptEngine.eval(expression, scriptContext);
    }

    private void advancedRepl(ConsoleReader consoleReader) throws Exception {
        evaluateBuiltins();
        for (; ; ) {
            final String line = consoleReader.readLine(getPrompt());
            final String trimmed = line != null ? line.trim() : null;
            if (trimmed == null || trimmed.startsWith("quit") || trimmed.startsWith("exit")) {
                exit();
                break;
            } else if (trimmed.startsWith("help")) {
                HelpScanner.printHelp();
                continue;
            }
            try {
                Object evaluationResult = evaluate(line);
                if (evaluationResult instanceof String) {
                    out.println((String) evaluationResult);
                } else if (evaluationResult != null) {
                    out.println(evaluationResult.getClass().getSimpleName() + ": " + evaluationResult);
                } else {
                    //FIXME: not sure what to give back; maybe out.println("Ok"); ?
                }
            } catch (ScriptException e) {
                printScriptException(e);
            }
        }
    }

    private void evaluateBuiltins() {
        try {
            evaluate("import __builtin__");
            cliObjectNames.forEach(cliObject -> {
                try {
                    evaluate(String.format("__builtin__.%s = %s", cliObject, cliObject));
                } catch (ScriptException e) {
                    printScriptException(e);
                }
            });
        } catch (ScriptException e) {
            printScriptException(e);
        }

    }

    private void printScriptException(ScriptException e) {
        Throwable cause = e.getCause();
        if (cause == null) {
            System.err.println(e);
        } else if (cause instanceof PyException) {
            if (cause.getCause() == null) {
                System.err.println(cause);
            } else if (cause.getCause() instanceof DeployitClientException) {
                printValidationMessages((DeployitClientException) cause.getCause());
            } else {
                System.err.println(cause.getCause());
            }
        } else {
            System.err.println(cause);
        }
    }

    private void printValidationMessages(DeployitClientException exception) {
        Object entity = exception.getEntity();
        if (entity instanceof ConfigurationItem) {
            printCiValidation((ConfigurationItem) entity);
        } else if (entity instanceof List) {
            ((List<ConfigurationItem>) entity).forEach(this::printCiValidation);
        } else {
            System.err.println(exception);
        }
    }

    private void printCiValidation(ConfigurationItem ci) {
        List<ValidationMessage> validationMessages = ci.get$validationMessages();
        if (!validationMessages.isEmpty()) {
            System.out.printf("Configuration item '%s' contained validation errors:%n", ci.getId());
            validationMessages.forEach(message -> System.out.printf("- property '%s': %s%n", message.getPropertyName(), message.getMessage()));
        }
    }

    protected void exit() {
        System.out.println("Bye bye!");
    }

    String getPrompt() {
        String username = communicator.getHttpClientHolder().getUserName();
        if (username != null) {
            return username + " > ";
        } else {
            return "deployit > ";
        }
    }

    public ScriptContext getScriptContext() {
        return scriptContext;
    }
}
