package com.xebialabs.deployit.cli;

import com.xebialabs.deployit.booter.remote.BooterConfig;
import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.booter.remote.RemoteBooter;
import com.xebialabs.deployit.cli.api.ProxiesInstance;
import com.xebialabs.deployit.cli.help.HelpScanner;
import com.xebialabs.deployit.cli.util.PasswordEncrypter;
import jline.Terminal;
import jline.TerminalFactory;
import jline.console.ConsoleReader;
import nl.javadude.scannit.Configuration;
import nl.javadude.scannit.Scannit;
import nl.javadude.scannit.scanner.TypeAnnotationScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.script.ScriptException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import static com.xebialabs.deployit.cli.CliOptions.parse;
import static java.util.stream.Collectors.toSet;

public class Cli {

    public static final String CLI_PASSWORD_PROPERTY_NAME = "cli.password";
    public static final String CLI_USERNAME_PROPERTY_NAME = "cli.username";

    private static final AtomicReference<Properties> properties = new AtomicReference<>();
    private final ConsoleReader consoleReader;
    private final ScriptEngineBuilder scriptEngine;
    private final CliOptions options;
    private DeployitCommunicator communicator;
    private PasswordEncrypter encrypter;
    private Set<String> cliObjectNames;

    public Cli(CliOptions options) throws Exception {
        this(options, null);
    }

    public Cli(CliOptions options, ConsoleReader consoleReader) throws Exception {
        this.options = options;
        this.consoleReader = consoleReader;
        scriptEngine = new ScriptEngineBuilder();
        initialize();
    }

    private void printBanner() {
        bannerPrint("Welcome to the XL Deploy Jython CLI!");
        bannerPrint("Type 'help' to learn about the objects you can use to interact with XL Deploy.");
        bannerPrint("");
    }

    private void initialize() throws Exception {
        PasswordEncrypter.init();
        encrypter = PasswordEncrypter.getInstance();

        BooterConfig.Builder builder = BooterConfig.builder().withHost(options.getHost()).withContext(options.getContext()).withPort(options.getPort())
                .withProxyHost(options.getProxyHost()).withProxyPort(options.getProxyPort()).withSocketTimeout(options.getSocketTimeout());
        if (options.isSecured()) {
            builder.withProtocol(BooterConfig.Protocol.HTTPS);
        }
        createCredentials(builder);
        if (consoleReader != null) {
            consoleReader.addCompleter(new CliCompleter(consoleReader));
        }
        BooterConfig config = builder.build();
        RemoteBooter.boot(config);
        communicator = RemoteBooter.getCommunicator(config);
        printBanner();
        ProxiesInstance proxies = createAndRegisterProxies(config);
        registerCliObjects(proxies);
    }

    private ProxiesInstance createAndRegisterProxies(BooterConfig config) {
        ProxiesInstance proxies = new ProxiesInstance(config);
        if (options.isExposeProxies()) {
            System.out.println("Exposing Proxies!");
            scriptEngine.put("proxies", proxies);
        }
        return proxies;
    }

    public static void main(String[] args) throws Exception {
        CliOptions options = parse(args);
        if (options == null) {
            return;
        }
        new Cli(options, setupConsole()).getNewInterpreter().interpret();
    }

    private static ConsoleReader setupConsole() throws IOException {
        final Terminal terminal = getTerminal();
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            try {
                terminal.restore();
            } catch (Exception ignored) {
            }
        }));
        ConsoleReader cr = new ConsoleReader();
        cr.setExpandEvents(false);
        return cr;
    }

    private static Terminal getTerminal() {
        Terminal terminal;
        try {
            terminal = TerminalFactory.get();
        } catch (Exception t) {
            System.err.println("[WARNING] Error loading terminal, using fallback. Your terminal will have reduced functionality. Please see: https://docs.xebialabs.com/deploy/how-to/troubleshooting-the-cli.html");
            logger.warn("Error loading terminal, using fallback. Your terminal will have reduced functionality.");
            TerminalFactory.configure(TerminalFactory.Type.NONE);
            terminal = TerminalFactory.get();
        }
        return terminal;
    }

    public Interpreter getNewInterpreter() throws ScriptException, IOException {
        final Interpreter interpreter = new Interpreter(consoleReader, scriptEngine, communicator, options, cliObjectNames);
        readExtensions(interpreter);
        return interpreter;
    }

    private void registerCliObjects(final ProxiesInstance proxies) throws Exception {
        final Scannit scannit = new Scannit(Configuration.config().with(new TypeAnnotationScanner()).scan("com.xebialabs"));
        final Set<Class<?>> classes = scannit.getTypesAnnotatedWith(CliObject.class);
        cliObjectNames = classes.stream().map(cliObject -> cliObject.getAnnotation(CliObject.class).name()).collect(toSet());
        for (Class<?> cliObject : classes) {
            final Constructor<?> constructor = cliObject.getConstructor(ProxiesInstance.class);
            final Object o = constructor.newInstance(proxies);
            final String name = cliObject.getAnnotation(CliObject.class).name();
            scriptEngine.put(name, o);
        }
        HelpScanner.init(classes);
        if (!options.isQuiet()) {
            HelpScanner.printHelp();
        }
    }

    public void createCredentials(BooterConfig.Builder builder) throws IOException {
        builder.withCredentials(retrieveUsername(), retrievePassword());
    }

    private String retrieveUsername() throws IOException {
        String username;
        if (options.isUsernameOnCommandline()) {
            username = options.getUsername();
        } else if (deployitConfigurationFileExists() && readFromProperties(CLI_USERNAME_PROPERTY_NAME) != null) {
            username = readFromProperties(CLI_USERNAME_PROPERTY_NAME);
        } else {
            username = consoleReader.readLine("Username: ");
        }
        return username;
    }

    private String retrievePassword() throws IOException {
        String password;
        if (options.isPasswordOnCommandline()) {
            password = options.getPassword();
        } else if (deployitConfigurationFileExists() && readFromProperties(CLI_PASSWORD_PROPERTY_NAME) != null) {
            password = encrypter.ensureDecrypted(readFromProperties(CLI_PASSWORD_PROPERTY_NAME));
        } else {
            password = consoleReader.readLine("Password: ", '\0');
        }
        return password;
    }

    private boolean deployitConfigurationFileExists() {
        return options.getConfigurationFile().exists();
    }

    private String readFromProperties(final String key) throws IOException {
        if (properties.get() == null) {
            Properties props = readDeployitConfigurationFile();
            properties.set(props);
        }

        return properties.get().getProperty(key);
    }

    private Properties readDeployitConfigurationFile() throws IOException {
        return readDeployitConfigurationFile(options, encrypter);
    }

    static Properties readDeployitConfigurationFile(CliOptions options, PasswordEncrypter encrypter) throws IOException {
        Properties props = new Properties();

        try (FileInputStream inStream = new FileInputStream(options.getConfigurationFile())) {
            props.load(inStream);
        }

        String cliPassword = props.getProperty(CLI_PASSWORD_PROPERTY_NAME);
        if (cliPassword != null) {
            cliPassword = encrypter.ensureEncrypted(cliPassword);
            props.setProperty(CLI_PASSWORD_PROPERTY_NAME, cliPassword);

            if (options.getConfigurationFile().canWrite()) {
                logger.info("Writing configuration file [{}] with encrypted " + CLI_PASSWORD_PROPERTY_NAME, options.getConfigurationFile());
                try (FileOutputStream outStream = new FileOutputStream(options.getConfigurationFile())) {
                    props.store(outStream, "XL Deploy CLI configuration file");
                } catch (IOException exc) {
                    logger.error("Error while updating configuration file [{}]: [{}]", options.getConfigurationFile(), exc.getMessage());
                }
                logger.info("Done writing configuration file [{}] with encrypted " + CLI_PASSWORD_PROPERTY_NAME, options.getConfigurationFile());
            } else {
                logger.warn("Cannot write configuration file [{}], not encrypting " + CLI_PASSWORD_PROPERTY_NAME, options.getConfigurationFile());
            }
        }

        return props;
    }

    private void readExtensions(Interpreter interpreter) throws ScriptException, IOException {
        final File extensionDir = new File(options.getExtensionFolderName());
        if (!extensionDir.exists() || !extensionDir.isDirectory()) {
            System.out.println("No extension directory present.");
            return;
        }

        List<File> files = Files.walk(Paths.get(options.getExtensionFolderName())).
                filter(Files::isRegularFile).
                filter((path) -> path.toString().endsWith(".py") || path.toString().endsWith(".cli")).
                map((Path path) -> path.toFile().getAbsoluteFile()).
                sorted().
                collect(Collectors.toList());

        for (File extension : files) {
            bannerPrint("Reading extension: " + extension);
            interpreter.evaluateFile(extension.getAbsolutePath());
        }
    }

    void bannerPrint(String line) {
        if (!options.isQuiet()) {
            System.out.println(line);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(Cli.class);
}
