package com.xebialabs.deployit.cli.api;

import com.xebialabs.deployit.booter.remote.DeployitCommunicator;
import com.xebialabs.deployit.booter.remote.client.DeployitRemoteClient;
import com.xebialabs.deployit.booter.remote.resteasy.DeployitClientException;
import com.xebialabs.deployit.cli.CliObject;
import com.xebialabs.deployit.cli.api.internal.DescriptorHelper;
import com.xebialabs.deployit.cli.api.internal.PrintHelper;
import com.xebialabs.deployit.cli.help.ClassHelp;
import com.xebialabs.deployit.cli.help.MethodHelp;
import com.xebialabs.deployit.cli.help.ParameterHelp;
import com.xebialabs.deployit.core.api.dto.Comparison;
import com.xebialabs.deployit.engine.api.TaskBlockService;
import com.xebialabs.deployit.engine.api.dto.Control;
import com.xebialabs.deployit.engine.api.dto.ServerInfo;
import com.xebialabs.deployit.engine.api.execution.*;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CountDownLatch;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.deployit.engine.api.execution.StepExecutionState.*;


/**
 * Main API class, exposing calls into XL Deploy.
 */
@CliObject(name = "deployit")
@ClassHelp(description = "The main gateway to interfacing with XL Deploy.")
public class DeployitClient extends DocumentedObject {

    private ProxiesInstance proxies;
    private DeployitCommunicator communicator;

    // Needed for tests
    public DeployitClient() {}

    public DeployitClient(ProxiesInstance proxies)  {
        this.proxies = proxies;
        this.communicator = proxies.getCommunicator();
    }

    @MethodHelp(description = "Import a package that is located on the server, on the local file system, or in an external location.", parameters = {
            @ParameterHelp(name = "importablePackage", description = "This is:\n\t- The name of the importable package on the server\n\t- The absolute path to a local importable package\n\t- The URI of an externally stored package")
    })
    public ConfigurationItem importPackage(final String importablePackageName) {
        return new DeployitRemoteClient(communicator).importPackage(importablePackageName);
    }

    @MethodHelp(description = "List all importable packages on the server.")
    public List<String> listImportablePackages() {
        return proxies.getImportablePackage().list();
    }

    @MethodHelp(description = "Gracefully stop an active deployment task", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.")
    })
    public void stopTask(String taskId) {
        proxies.getTaskBlockRegistry().stop(taskId);
    }

    @MethodHelp(description = "Start a deployment task", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.")
    })
    public void startTask(String taskId) {
        startTask(taskId, proxies);
    }

    @MethodHelp(description = "Start a deployment task and wait for it to finish", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.")
    })
    public void startTaskAndWait(final String taskId) {
        startTaskAndWait(taskId, proxies, false);
    }

    public void waitForTask(String taskId) {
        waitForTask(taskId, proxies, false);
    }

    @MethodHelp(description = "Abort an active deployment task", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.")
    })
    public void abortTask(String taskId) {
        proxies.getTaskBlockRegistry().abort(taskId);
    }

    @MethodHelp(description = "Cancel a stopped deployment task", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task.")
    })
    public void cancelTask(String taskId) {
        proxies.getTaskBlockRegistry().cancel(taskId);
    }

    public void skipSteps(String taskId, List<Integer> stepIds) {
        if (stepIds == null || stepIds.isEmpty()) {
            return;
        }

        // Get current step states
        TaskWithSteps steps = proxies.getTaskRegistry().getSteps(taskId);

        // Determine which steps should be skipped and which should be unskipped
        List<Integer> skip = new ArrayList<>();
        List<Integer> unskip = new ArrayList<>();
        for (Integer stepId : stepIds) {
            StepExecutionState state = steps.getSteps().get(stepId - 1).getState();
            if (state == PENDING || state == FAILED) {
                skip.add(stepId);
            } else if (state == SKIP) {
                unskip.add(stepId);
            }
        }

        // Skip and unskip
        if (!skip.isEmpty()) {
            proxies.getTaskRegistry().skip(taskId, skip);
        }
        if (!unskip.isEmpty()) {
            proxies.getTaskRegistry().unskip(taskId, unskip);
        }
    }

    @MethodHelp(description = "(Un)skip steps of the task.\n- If a step is in the PENDING or FAILED state, it will be SKIPPED.\n- If a step is in the SKIPPED state it will be PENDING. ", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task"),
            @ParameterHelp(name = "stepIds", description = "The ids of the steps to skip or unskip")
    })
    public void skipSteps(String taskId, Integer[] stepIds) {
        if (stepIds == null) {
            return;
        }

        skipSteps(taskId, newArrayList(stepIds));
    }

    @MethodHelp(description = "Assign a pending task to another principal", parameters = {
            @ParameterHelp(name = "taskId", description = "The id of the task."),
            @ParameterHelp(name = "owner", description = "The new task owner.")
    })
    public void assignTask(String taskId, String owner) {
        proxies.getTaskBlockRegistry().assign(taskId, owner);
    }

    @MethodHelp(description = "Creates a task to discover middleware starting with given CI. Refer to plugin configuration item documentation to see which fields are required for discover.", parameters = {
            @ParameterHelp(name = "ci", description = "The configuration item with all required discovery fields filled in.")
    })
    public String createDiscoveryTask(ConfigurationItem ci) {
        return proxies.getDiscovery().createTask(proxies.getDiscovery().prepare(ci));
    }


    @MethodHelp(description = "Returns results for the already executed discovery task.", parameters = {
            @ParameterHelp(name = "taskId", description = "Id of the executed discovery task.")
    })
    public List<ConfigurationItem> retrieveDiscoveryResults(String taskId) {
        return proxies.getDiscovery().retrieveInspectionResults(taskId);
    }

    @MethodHelp(description = "Prepares a control task object which can be used to set parameters and have control over the execution of the control task.", parameters = {
            @ParameterHelp(name = "ci", description = "The configuration item to execute the control task on"),
            @ParameterHelp(name = "taskName", description = "The control task name eg. \"start\"")
    })
    public Control prepareControlTask(ConfigurationItem ci, String taskName) {
        Control prepare = proxies.getControlTask().prepare(taskName, ci.getId());
        logger.info("Control[params] = " + prepare.getParameters());
        logger.info("Control[ci] = " + prepare.getConfigurationItem());
        return prepare;

    }

    @MethodHelp(description = "Create the prepared control task and returns the task id. Needs to be manually executed using the task service.", parameters = {
            @ParameterHelp(name = "control", description = "The prepared control task object")
    })
    public String createControlTask(Control control) {
        return proxies.getControlTask().createTask(control);
    }

    @MethodHelp(description = "Execute the control task on the provided configuration item. If the control tasks executes successfully, no output is shown. In the case of an error, log output is shown.", parameters = {
            @ParameterHelp(name = "taskName", description = "The control task name eg. \"start\""),
            @ParameterHelp(name = "ci", description = "The configuration item to execute the control task on")
    })
    public void executeControlTask(String taskName, ConfigurationItem ci) {
        Control control = proxies.getControlTask().prepare(taskName, ci.getId());
        String task = proxies.getControlTask().createTask(control);
        startTaskAndWait(task);
    }

    @MethodHelp(description = "Describe the CI class, with all the values it takes.", parameters = {
            @ParameterHelp(name = "shortName", description = "The (Short) name of the CI eg. \"Host\", \"WasDataSource\", etc")
    })
    public void describe(String typeName) {
        DescriptorHelper.describe(typeName);
    }

    @MethodHelp(description = "Print a tree-view of a CI", parameters = {
            @ParameterHelp(name = "ci", description = "The CI to print")
    })
    public void print(ConfigurationItem ci) {
        PrintHelper.getInstance().printCi(ci, proxies.getRepository());
    }

    public void print(Collection<ConfigurationItem> cis) {
        PrintHelper.getInstance().printCis(cis, proxies.getRepository());
    }

    public Comparison compare(String reference, List<String> ids) {
        return proxies.getComparison().compare(reference, ids);
    }

    @MethodHelp(description = "Retrieve XL Deploy server information")
    public ServerInfo info() {
        return proxies.getServer().getInfo();
    }

    @MethodHelp(description = "Run the XL Deploy server garbage collector")
    public void runGarbageCollector() {
        proxies.getServer().gc();
    }

    @MethodHelp(description = "Shutdown the XL Deploy server")
    public void shutdown() {
        proxies.getServer().shutdown();
    }

    protected static void startTask(String taskId, ProxiesInstance proxies) {
        proxies.getTaskBlockRegistry().start(taskId);
    }

    protected static void startTaskAndWait(final String taskId, ProxiesInstance proxies, boolean cancelIfFailed) {
        logger.debug("Starting and waiting for task {}", taskId);
        startTask(taskId, proxies);
        waitForTask(taskId, proxies, cancelIfFailed);
    }

    protected static void waitForTask(String taskId, ProxiesInstance proxies, boolean cancelIfFailed) {
        // Wait until done/failed
        final long sleepTime = 1000;
        boolean done = false;
        int retryCount = 0;
        TaskWithBlock taskSummary = null;
        while (!done) {
            try {
                taskSummary = proxies.getTaskBlockRegistry().getTaskSummary(taskId);
                final TaskExecutionState st = taskSummary.getState();
                if (st.isPassiveAfterExecuting()) {
                    logger.debug("Task {} passive with state {}", taskId, st);
                    done = true;
                } else {
                    try {
                        retryCount = 0;
                        logger.debug("Task {} still active {}, sleeping", taskId, st);
                        Thread.sleep(sleepTime);
                    } catch (InterruptedException ie) {
                        logger.debug("Task {} still active {}, interrupted", taskId, st);
                        Thread.currentThread().interrupt();
                    }
                }
            } catch (DeployitClientException e) {
                retryCount++;
                if (retryCount > 5) {
                    throw e;
                }
                try {
                    logger.debug("Task {} in waiting for state", taskId);
                    Thread.sleep(sleepTime);
                } catch (InterruptedException ie) {
                    logger.debug("Task {} in waiting for state, interrupted", taskId);
                    Thread.currentThread().interrupt();
                }
            }
        }
        TaskExecutionState state = taskSummary.getState();
        if (state == TaskExecutionState.EXECUTED) {
            logger.debug("Task {} passive with state {}, archiving....", taskId, state);
            archiveAndWait(taskId, proxies);
        } else if (cancelIfFailed && state.isExecutionHalted()) {
            logger.debug("Task {} passive with state {}, canceling....", taskId, state);
            proxies.getTaskBlockRegistry().cancel(taskId);
        }
        logger.debug("Task {} has finished waiting with state {}", taskId, state);
    }

    // this method make sure the task state transition between EXECUTED -> DONE is finished
    private static void archiveAndWait(String taskId, ProxiesInstance proxies) {
        TaskBlockService taskRegistry = proxies.getTaskBlockRegistry();
        logger.debug("Archiving task {}", taskId);
        taskRegistry.archive(taskId);
        boolean archived = false;
        int tries = 0;
        int sleepTime = 500; // ms
        int timeout = 30000; // ms
        CountDownLatch latch = new CountDownLatch(5);
        while (!archived && tries < timeout / sleepTime && latch.getCount() > 0) {
            try {
                TaskWithBlock taskSummary = taskRegistry.getTaskSummary(taskId);
                final TaskExecutionState st = taskSummary.getState();
                if (st.isFinal()) {
                    logger.debug("Task {} archiving complete with state {}", taskId, st);
                    archived = true;
                } else {
                    sleepForAwhile(taskId, st, sleepTime);
                    tries++;
                }
            }
            catch (DeployitClientException e) {
                latch.countDown();
                sleepForAwhile(taskId, null, sleepTime);
                tries++;
            }
        }
        if (!archived) {
            throw new IllegalStateException("Timeout waiting for task " + taskId + " to archive after " + timeout + " ms.");
        } else {
            logger.debug("Task {} done archiving", taskId);
        }
    }

    private static void sleepForAwhile(String taskId, TaskExecutionState st, int sleepTime) {
        try {
            logger.debug("Task {} still not archived {}, sleeping", taskId, st);
            Thread.sleep(sleepTime);
        } catch (InterruptedException e) {
            logger.debug("Task {} still not archived {} but interrupted", taskId, st);
            Thread.currentThread().interrupt();
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployitClient.class);
}
