package com.xebialabs.deployit.client;

import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.xebialabs.deployit.client.logger.Slf4jDeploymentListener;
import com.xebialabs.deployit.client.ssl.SelfSignedCertificateAcceptingSocketFactory;
import com.xebialabs.deployit.core.api.dto.*;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.protocol.Protocol;
import org.apache.commons.httpclient.protocol.ProtocolSocketFactory;
import org.apache.commons.lang.StringUtils;

import javax.ws.rs.core.Response;
import java.io.IOException;
import java.net.MalformedURLException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.ImmutableList.copyOf;
import static com.google.common.collect.Iterables.any;
import static com.google.common.collect.Iterators.filter;
import static java.lang.String.format;

/**
 * Client for Deployit.
 */
public class DeployitCli {

	private final DeploymentListener listener;

	private ConnectionOptions options;
	private final Authentication authentication;

	private final DeployitProxies proxies;

	private final DeploymentClient deploymentClient;
	private final DeployitClient deployitClient;
	private final RepositoryClient repositoryClient;

	private AtomicReference<Descriptors> descriptors = new AtomicReference<Descriptors>();

	public DeployitCli(ConnectionOptions options) {
		this(options, null);
	}

	public DeployitCli(ConnectionOptions options, DeploymentListener listener) {
		this.listener = (listener != null ? listener : new Slf4jDeploymentListener(DeployitCli.class));
		this.options = options;
		this.authentication = new Authentication(options);
		try {
			setupSecureCommunications();
			attemptToConnectToServer();
			proxies = new DeployitProxies(options, authentication);
			deploymentClient = new DeploymentClient(proxies);
			deployitClient = new DeployitClient(proxies);
			repositoryClient = new RepositoryClient(proxies);
		} catch (Exception e) {
			throw new RuntimeException("Initialization failed", e);
		}
	}

	public void resetCredentials(String username, String password) {
		if (authentication.getUserName().equals(username) && authentication.password.equals(password))
			return;

		authentication.logout();
		authentication.loginAs(username, password);
	}

	private void setupSecureCommunications() {
		if (options.isSecured()) {
			Protocol easyhttps = new Protocol("https", (ProtocolSocketFactory) new SelfSignedCertificateAcceptingSocketFactory(), 443);
			Protocol.registerProtocol("https", easyhttps);
		}
	}

	private void attemptToConnectToServer() {
		String urlToConnectTo = options.getUrl();
		getListener().info("Connecting to the Deployit server at " + urlToConnectTo + "...");
		try {
			final int responseCode = authentication.getHttpClient().executeMethod(new GetMethod(urlToConnectTo + "/server/info"));
			if (responseCode == 200) {
				getListener().info("Succesfully connected.");
			} else if (responseCode == 401 || responseCode == 403) {
				throw new IllegalStateException("You were not authenticated correctly, did you use the correct credentials?");
			} else {
				throw new IllegalStateException("Could contact the server at " + urlToConnectTo + " but received an HTTP error code, " + responseCode);
			}
		} catch (MalformedURLException mue) {
			throw new IllegalStateException("Could not contact the server at " + urlToConnectTo, mue);
		} catch (IOException e) {
			throw new IllegalStateException("Could not contact the server at " + urlToConnectTo, e);
		}
	}

	public RepositoryObject create(Container ci) {
		final String id = ci.getId();
		try {
			return get(id);
		} catch (Exception e) {
			getListener().debug(format("%s does not exist, create it", id));
			ConfigurationItemDto configurationItem = new ConfigurationItemDto(id, ci.getType());
			configurationItem.setValues(Maps.<String, Object>newHashMap(ci.getProperties()));
			final Response response = getProxies().getRepository().create(configurationItem.getId(), configurationItem);
			return checkForValidations(response);
		}
	}

	public void delete(String id) {
		try {
			getListener().debug("Delete " + id);
			repositoryClient.delete(id);
		} catch (Exception e) {
			getListener().debug(format("delete fails %s", id));
		}
	}

	public List<String> search(String type) {
		try {
			getListener().debug("search " + type);
			return repositoryClient.search(type, null);
		} catch (Exception e) {
			getListener().debug(format("search fails for %s %s", type, e.getMessage()));
		}
		return Collections.emptyList();
	}

	public ServerInfo info() {
		final Response response = getProxies().getServer().getInfo();
		return new ResponseExtractor(response).getEntity();
	}

	public RepositoryObject get(String ciId) {
		final Response response = getProxies().getRepository().read(ciId);
		return checkForValidations(response);
	}

	public Descriptors getDescriptors() {
		if (descriptors.get() == null) {
			synchronized (this) {
				final Response response = getProxies().getDescriptor().list();
				final ResponseExtractor responseExtractor = new ResponseExtractor(response);
				final ConfigurationItemDescriptorList entity = responseExtractor.getEntity();
				descriptors.set(new Descriptors(entity.getDescriptors()));
			}
		}
		return descriptors.get();
	}

	public RepositoryObject importPackage(String darFileLocation, DeploymentListener listener) {
		getListener().setActionListener(listener);
		try {
			return deployitClient.importPackage(darFileLocation);
		} finally {
			getListener().setActionListener(null);
		}
	}

	public String deploy(String source, String target, List<? extends Deployed> configuredDeployeds, DeploymentOptions deploymentOptions, DeploymentListener listener) {
		getListener().setActionListener(listener);

		getListener().debug(deploymentOptions.toString());

		Deployment deployment;
		String previouslyDeployedApplicationId = null;
		boolean initialDeployment = isInitialDeployment(source, target);
		if (initialDeployment) {
			getListener().info("initial Deployment");
			deployment = deploymentClient.prepareInitial(source, target);
			if (!deploymentOptions.isExplicitMappings()) {
				getListener().debug(" generateAllDeployeds");
				deployment = deploymentClient.generateAllDeployeds(deployment);
			}
		} else {
			getListener().info("upgrade Deployment");
			String deployedApplicationId = getDeployedApplicationId(source, target);
			previouslyDeployedApplicationId = getPreviousDeployedPackage(deployedApplicationId);
			deployment = deploymentClient.prepareUpgrade(source, deployedApplicationId);
			if (!deploymentOptions.isExplicitMappings() && deploymentOptions.isGenerateDeployedOnUpgrade()) {
				getListener().debug(" generateAllDeployeds");
				deployment = deploymentClient.generateAllDeployeds(deployment);
			}
		}

		if (deploymentOptions.isExplicitMappings()) {
			getListener().debug("use explicits deployeds");
			for (Deployed configuredDeployed : configuredDeployeds) {
				checkNotNull(configuredDeployed.getId(), "id is mandatory in the explicit deployed mode");
				String type = configuredDeployed.getType();
				checkNotNull(type, "type is mandatory in the explicit deployed mode");

				final String deployableId = configuredDeployed.getDeployable(source);
				final String containerId = configuredDeployed.getContainer();
				final boolean exist = any(deployment.getDeployeds(), new Predicate<ConfigurationItemDto>() {
					@Override
					public boolean apply(ConfigurationItemDto input) {
						return input.getValues().get("deployable").equals(deployableId)
								&& input.getValues().get("container").equals(containerId);
					}
				});
				if (exist) {
					getListener().debug(format(" deployed  %s %s %s already exists", deployableId, containerId, type));
				} else {
					getListener().debug(format(" generateSingleDeployed %s %s %s", deployableId, containerId, type));
					deployment = deploymentClient.generateSingleDeployed(deployableId, containerId, type, deployment);
				}
			}
		}

		if (configuredDeployeds != null) {
			getListener().debug("update the generated deployeds with the configured deployeds");
			final List<ConfigurationItemDto> deployeds = deployment.getDeployeds();
			for (ConfigurationItemDto configurationItem : deployeds) {
				getListener().debug(" check " + configurationItem);
				final String id = configurationItem.getId();
				final Iterator<? extends Deployed> deployedIterator = filter(configuredDeployeds.iterator(), new Predicate<Deployed>() {
					@Override
					public boolean apply(Deployed input) {
						return input.getId().equals(id);
					}
				});
				if (deployedIterator.hasNext()) {
					updateConfigurationItemValues(configurationItem, deployedIterator.next());
				} else {
					getListener().debug(" no configured deployed found with id " + configurationItem.getId());
				}
			}
		}

		getListener().debug(" dump Deployeds");
		for (ConfigurationItemDto itemDto : deployment.getDeployeds()) {
			getListener().debug(" - " + itemDto);
		}

		try {
			getListener().debug("validate");
			deployment = deploymentClient.validate(deployment);
		} catch (RuntimeException e) {
			getListener().error(" RuntimeException: " + e.getMessage());
			if (deploymentOptions.isFailIfNoStepsAreGenerated() || !e.getMessage().contains("The task did not deliver any steps")) {
				throw e;
			}
			return null;
		}

		int validationMessagesFound = 0;
		for (ConfigurationItemDto configurationItem : deployment.getDeployeds()) {
			for (Message msg : configurationItem.getValidations()) {
				getListener().error(format("Validation error found on '%s' on field '%s': %s, %s", configurationItem.getId(), msg.getField(), msg.getMessage(), configurationItem));
				getListener().error(format(" %s", configurationItem));
				validationMessagesFound++;
			}
		}

		if (validationMessagesFound > 0) {
			throw new IllegalStateException(format("Validation errors (%d) have been found", validationMessagesFound));
		}

		getListener().debug("deploy");
		executeTask(deploymentClient.deploy(deployment).getTaskId(), deploymentOptions);

		if (shouldDeletePreviousVersion(deploymentOptions, previouslyDeployedApplicationId)) {
			getListener().info("Delete previously deployed dar " + previouslyDeployedApplicationId);
			delete(previouslyDeployedApplicationId);
		}
		return previouslyDeployedApplicationId;
	}

	public void undeployAndWait(String source) {
		getListener().info("   undeployAndWait " + source);
		String taskId = this.deploymentClient.undeploy(source).getTaskId();
		executeTask(taskId, new DeploymentOptions());
	}


	private RepositoryObject checkForValidations(final Response response) {
		final ResponseExtractor responseExtractor = new ResponseExtractor(response);
		final RepositoryObject ci = responseExtractor.getEntity();
		if (!responseExtractor.isValidResponse() && !ci.getValidations().isEmpty()) {
			throw new IllegalStateException(format("Configuration item contained validation errors: {%s}", ci.getValidations()));
		}
		return ci;
	}

	private void updateConfigurationItemValues(ConfigurationItemDto configurationItem, Deployed configuredDeployed) {
		getListener().debug(format(" update values of %s with %s", configuredDeployed.getId(), configuredDeployed.getValues()));
		configurationItem.getValues().putAll(configuredDeployed.getValues());
		if (!configuredDeployed.getPlaceholders().isEmpty()) {
			if (!configurationItem.getValues().containsKey("placeholders")) {
				configurationItem.getValues().put("placeholders", new HashMap<String, String>());
			}
			Map<String, String> placeholders = (Map<String, String>) configurationItem.getValues().get("placeholders");
			placeholders.putAll(configuredDeployed.getPlaceholders());
		}
		getListener().debug(configurationItem.getValues().toString());
	}


	private boolean executeTask(String taskId, DeploymentOptions deploymentOptions) {
		if (deploymentOptions.isSkipMode()) {
			getListener().info("skip mode, skip all the steps");
			getDeployitClient().skipSteps(taskId, range(getDeployitClient().retrieveTaskInfo(taskId).getNrOfSteps() + 1));
		}
		checkTaskState(taskId);
		if (deploymentOptions.isTestMode()) {
			getListener().info("test mode, cancel task " + taskId);
			getDeployitClient().cancelTask(taskId);
			return false;
		}

		try {
			getListener().info("Start deployment task " + taskId);
			getDeployitClient().startTaskAndWait(taskId);
			checkTaskState(taskId);
			return true;
		} catch (RuntimeException e) {
			if (taskId != null) {
				getListener().error(format("Error when executing task %s: %s", taskId, e.getMessage()));
				if (deploymentOptions.isCancelTaskOnError())
					getDeployitClient().cancelTask(taskId);
			}
			throw e;
		}

	}

	private boolean isInitialDeployment(String source, String target) {
		String deployedApplicationId = getDeployedApplicationId(source, target);
		getListener().debug("  deployedApplicationId " + deployedApplicationId);
		try {
			get(deployedApplicationId);
			return false;
		} catch (Exception e) {
			return true;
		}
	}


	private String getDeployedApplicationId(String source, String target) {
		//target = "Environments/DefaultEnvironment/deployit-petclinic"
		return Joiner.on("/").join(target, copyOf(Splitter.on("/").split(source)).get(1));
	}

	private String getPreviousDeployedPackage(String target) {
		final RepositoryObject dp = repositoryClient.read(target);
		final Map<String, Object> values = dp.getValues();
		Object source = values.get("version");
		return (source == null ? null : source.toString());
	}

	private void checkTaskState(String taskId) {
		final TaskInfo taskInfo = getDeployitClient().retrieveTaskInfo(taskId);
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd hh:mm:ss");
		getListener().info(format("%s Label      %s", taskId, taskInfo.getLabel()));
		getListener().info(format("%s State      %s %d/%d", taskId, taskInfo.getState(), taskInfo.getCurrentStepNr(), taskInfo.getNrOfSteps()));
		if (taskInfo.getStartDate() != null) {
			final GregorianCalendar startDate = (GregorianCalendar) taskInfo.getStartDate();
			getListener().info(format("%s Start      %s", taskId, sdf.format(startDate.getTime())));
		}

		if (taskInfo.getCompletionDate() != null) {
			final GregorianCalendar completionDate = (GregorianCalendar) taskInfo.getCompletionDate();
			getListener().info(format("%s Completion %s", taskId, sdf.format(completionDate.getTime())));
		}

		StringBuilder sb = new StringBuilder();
		for (int i = 1; i <= taskInfo.getNrOfSteps(); i++) {
			final Response stepInfoResponse = getProxies().getTaskRegistry().getStepInfo(taskId, i, null);
			final StepInfo stepInfo = new ResponseExtractor(stepInfoResponse).getEntity();
			final String description = stepInfo.getDescription();
			final String log = stepInfo.getLog();
			String stepInfoMessage;
			if (StringUtils.isEmpty(log) || description.equals(log)) {
				stepInfoMessage = format("%s step #%d %s\t%s", taskId, i, stepInfo.getState(), description);
			} else {
				stepInfoMessage = format("%s step #%d %s\t%s\n%s", taskId, i, stepInfo.getState(), description, log);
			}

			getListener().info(stepInfoMessage);
			if ("FAILED".endsWith(stepInfo.getState()))
				sb.append(stepInfoMessage);
		}

		if ("STOPPED".equals(taskInfo.getState()))
			throw new IllegalStateException(format("Errors when executing task %s: %s", taskId, sb));
	}


	private Integer[] range(int end) {
		Integer[] result = new Integer[end - 1];
		for (int i = 1; i < end; i++) {
			result[i - 1] = i;
		}
		return result;
	}

	private boolean shouldDeletePreviousVersion(DeploymentOptions deploymentOptions, String previousPackageId) {
		if (deploymentOptions.isTestMode())
			return false;
		return deploymentOptions.isDeletePreviouslyDeployedArtifact() && Strings.emptyToNull(previousPackageId) != null;
	}


	private DeployitClient getDeployitClient() {
		return deployitClient;
	}

	private DeploymentClient getDeploymentClient() {
		return deploymentClient;
	}

	private DeployitProxies getProxies() {
		return proxies;
	}

	public DeploymentListener getListener() {
		return listener;
	}
}
