/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.engine.api;

import java.util.List;
import java.util.Map;

import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.engine.api.dto.SelectedDeployment;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.api.execution.TaskPreviewBlock;
import com.xebialabs.xlplatform.documentation.PublicApi;
import com.xebialabs.deployit.plugin.api.reflect.Type;

/**
 * Deployment management
 */
@Path("/deployment")
@Consumes({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@Produces({ MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON })
@PublicApi
public interface DeploymentService {

    /**
     * Returns whether or not a {@link com.xebialabs.deployit.plugin.api.udm.Version udm.Version} of the
     * {@link com.xebialabs.deployit.plugin.api.udm.Application udm.Application} that is passed in is already deployed to the
     * {@link com.xebialabs.deployit.plugin.api.udm.Environment udm.Environment}.
     *
     * @param applicationId The ID of the {@link com.xebialabs.deployit.plugin.api.udm.Application udm.Application}.
     * @param environmentId The ID of the {@link com.xebialabs.deployit.plugin.api.udm.Environment udm.Environment}.
     * @return <code>true</code> if there exists a deployed version of the application on the environment, <code>false</code> otherwise.
     */
    @GET
    @Path("exists")
    boolean isDeployed(@QueryParam("application") String applicationId, @QueryParam("environment") String environmentId);

    /**
     * Prepares an initial deployment.
     *
     * @permission deploy#initial
     * @param versionId     The ID of the {@link com.xebialabs.deployit.plugin.api.udm.Version udm.Version} that is the source of the deployment.
     * @param environmentId The ID of the {@link com.xebialabs.deployit.plugin.api.udm.Environment udm.Environment} that is the target of the deployment.
     * @return A new <code>Deployment</code> object to which you can add deployeds.
     */
    @GET
    @Path("prepare/initial")
    Deployment prepareInitial(@QueryParam("version") String versionId, @QueryParam("environment") String environmentId);

    /**
     * Prepares an update deployment.
     *
     * @permission deploy#upgrade
     * @param versionId             The ID of the new {@link com.xebialabs.deployit.plugin.api.udm.Version udm.Version} that is the source of the deployment.
     * @param deployedApplicationId The ID of the {@link com.xebialabs.deployit.plugin.api.udm.DeployedApplication udm.DeployedApplication} that is to be updated.
     * @return A new <code>Deployment</code> object which contains the updated deployeds.
     */
    @GET
    @Path("prepare/update")
    Deployment prepareUpdate(@QueryParam("version") String versionId, @QueryParam("deployedApplication") String deployedApplicationId);

    /**
     * Prepares an undeployment.
     *
     * @permission deploy#undeploy
     * @param deployedApplicationId The ID of the {@link com.xebialabs.deployit.plugin.api.udm.DeployedApplication udm.DeployedApplication} that is to be undeployed.
     * @return A new <code>Deployment</code> object which contains no deployeds (ie. all deployeds of the previous deployment will be deleted).
     */
    @GET
    @Path("prepare/undeploy")
    Deployment prepareUndeploy(@QueryParam("deployedApplication") String deployedApplicationId);

    /**
     * Prepares all the deployeds for the given deployment.
     *
     * This will keep any previous deployeds present in the deployment object that are already present, unless they cannot be deployed with regards to their tags.
     * It will add all the deployeds that are still missing.
     * <p/>
     * Also filters out the deployeds that do not have any source attached anymore (deployables that were previously present).
     * <p/>
     *
     * @permission deploy#initial
     * @param deployment    The prepared Deployment parameter object
     * @return An updated Deployment parameter object.
     */
    @POST
    @Path("prepare/deployeds")
    Deployment prepareAutoDeployeds(Deployment deployment);

    /**
     * Generates only the deployeds for the given deployables in the deployment.
     * <p/>
     * Try to generate deployeds from each of the deployables to all the matching {@link com.xebialabs.deployit.plugin.api.udm.Container Containers}
     * in the {@link com.xebialabs.deployit.plugin.api.udm.Environment Environment} of the Deployment parameter object.
     * <p/>
     *
     * @permission deploy#initial
     * @param deployableIds The list of IDs of deployables to generate the deployeds for.
     * @param deployment    The prepared Deployment parameter object
     * @return An updated Deployment parameter object.
     * @deprecated Use {@link DeploymentService#generateSelectedDeployeds(com.xebialabs.deployit.engine.api.dto.SelectedDeployment)}
     *
     */
    @POST
    @Path("generate/selected")
    @Deprecated
    Deployment generateSelectedDeployeds(@QueryParam("deployables") List<String> deployableIds, Deployment deployment);

    /**
     * Generates only the deployeds for the given deployables in the deployment.
     * <p/>
     * Try to generate deployeds from each of the deployables to all the matching {@link com.xebialabs.deployit.plugin.api.udm.Container Containers}
     * in the {@link com.xebialabs.deployit.plugin.api.udm.Environment Environment} of the Deployment parameter object.
     * <p/>
     * This request is same as <a href="com.xebialabs.deployit.engine.api.DeploymentService.html#/deployment/generate/selected:POST">/deployment/generate/selected</a> but does
     * not use URL query parameters avoiding it becoming too long.
     * <p/>
     *
     * @permission deploy#initial
     * @param selectedDeployment The prepared Deployment parameter object with list of IDs of deployables to generate the deployeds for.
     * @return An updated Deployment parameter object.
     */
    @POST
    @Path("generate/selected/deployables")
    Deployment generateSelectedDeployeds(SelectedDeployment selectedDeployment);

    /**
     * Generates a deployed for the given deployable to the given container in the deployment.
     * <p/>
     * The deployed should be of the given type (which is optional)
     * <p/>
     *
     * @permission deploy#initial
     * @param deployableId The ID of the deployable to generate a deployed for
     * @param containerId  The ID of the container to generate a deployed to
     * @param deployedType (Optional) The type of the deployed to generate
     * @param deployment    The prepared Deployment parameter object
     * @return An updated Deployment parameter object.
     */
    @POST
    @Path("generate/single")
    Deployment generateSingleDeployed(@QueryParam("deployable") String deployableId, @QueryParam("container") String containerId, @QueryParam("deployedtype") Type deployedType, Deployment deployment);

    /**
     * Validates the generated deployment.
     *
     * Checks whether all the deployeds that are in the deployment are valid.
     *
     * @param deployment The deployment to validate.
     * @return The validated deployment.
     */
    @POST
    @Path("validate")
    Deployment validate(Deployment deployment);

    /**
     * Calculate the plan that XL Deploy will execute for the given deployment. A plan consists of one or more nested
     * blocks. Blocks can be sequential, parallel, or a block containing steps.
     *
     * <em>Note:</em> The block returned is not a task yet, and as such can not be skipped or re-ordered.
     *
     * @permission deploy#initial for initial deployments
     * @permission deploy#upgrade for upgrades
     * @permission deploy#remove for undeployments
     * @param deployment
     *            The deployment to generate the block for.
     * @return The block.
     */
    @POST
    @Path("previewblock")
    TaskPreviewBlock taskPreviewBlock(Deployment deployment);

    /**
     * Get details on a step in a step block.
     *
     * @permission deploy#initial for initial deployments
     * @permission deploy#upgrade for upgrades
     * @permission deploy#remove for undeployments
     * @permission task#preview_step to see all details about a step
     *
     * @param deployment
     *            The deployment to generate the block for.
     * @param blockId
     *            The id of the step block to query the steps for.
     * @param stepNr
     *            The number of the step to retrieve.
     * @return The step.
     */
    @POST
    @Path("previewblock/{blockId}/{stepNr}")
    StepState taskPreviewBlock(Deployment deployment, @PathParam("blockId") String blockId, @PathParam("stepNr") int stepNr);

    /**
     * Creates the deployment task.
     *
     * @permission deploy#initial for initial deployments
     * @permission deploy#upgrade for upgrades
     * @permission deploy#remove for undeployments
     * @param deployment The fully prepared Deployment parameter object.
     * @return a reference to a Task ID that can be executed by the {@link TaskService}.
     */
    @POST
    String createTask(Deployment deployment);

    /**
     * Rollback a {@code STOPPED} or {@code EXECUTED} task. Reverting the deployment to the previous state. The task
     * will be set to {@code CANCELLED} when it was {@code STOPPED}, and {@code DONE} when it was {@code EXECUTED}.
     *
     * @param taskId the ID of the task
     * @return the ID of the new task.
     */
    @POST
    @Path("rollback/{taskid}")
    String rollback(@PathParam("taskid") String taskId);

    /**
     * Get the effective dictionary for an environment, optionally filtered by application and container.
     *
     * This will return the dictionary that is used when deploying to a specific environment. If an application/applicationVersion is passed, the dictionary is filtered to apply only to deployments of that application.
     * If a container is passed, the dictionary is filtered to apply only to deployeds pointing to that container in the environment.
     *
     * @param environment The ID of the environment
     * @param applicationVersion The ID of the applicationVersion (package) (optional, can be null)
     * @param application The ID of the application (optional, can be null). This parameter will be ignored if the applicationVersion is provided.
     * @param container The ID of the container (optional, can be null)
     *
     * @return the entries of the dictionary as it applies to a deployment to that environment
     */
    @GET
    @Path("dictionary")
    Map<String, String> effectiveDictionary(@QueryParam("environment") String environment, @QueryParam("applicationVersion") String applicationVersion, @QueryParam("application") String application, @QueryParam("container") String container);

}

