/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization.json;

import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;
import org.joda.time.DateTime;

import com.xebialabs.deployit.engine.api.execution.StepExecutionState;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.xltype.serialization.xstream.DateTimeAdapter;

import java.util.ArrayList;
import java.util.List;

public abstract class StepJsonConverter {

    private final DateTimeAdapter timeAdapter = new DateTimeAdapter();

    public String toJson(StepState step) {
        JsonWriter writer = new JsonWriter();

        writeJson(step, writer);

        return writer.toString();
    }

    public JsonWriter writeJson(StepState step, JsonWriter writer) {
        StepExecutionState state = step.getState();
        writer.object();
        writer.key("state").value(state);
        writer.key("failures").value(step.getFailureCount());
        writer.key("skippable").value(step.getSkippable());
        writer.key("description").value(step.getDescription());
        if (step.getStartDate() != null) {
            writer.key("startDate").value(timeAdapter.marshal(step.getStartDate()));
        }
        if (step.getCompletionDate() != null) {
            writer.key("completionDate").value(timeAdapter.marshal(step.getCompletionDate()));
        }
        if (step.getLog() != null) {
            writer.key("log").value(step.getLog());
        }
        if (step.getPreviousAttemptsLogs() != null) {
            writer.key("previousAttemptsLogs");
            writer.array();
            step.getPreviousAttemptsLogs().forEach(writer::value);
            writer.endArray();
        }
        writer.endObject();

        return writer;
    }

    public StepState toStep(String jsonString) {
        try {
            JSONObject json = new JSONObject(jsonString);

            StepExecutionState state = StepExecutionState.valueOf(json.getString("state"));
            int failureCount = json.getInt("failures");
            String description = json.getString("description");
            boolean skippable = json.getBoolean("skippable");
            DateTime startDate = null;
            if (json.has("startDate")) {
                startDate = timeAdapter.unmarshal(json.getString("startDate"));
            }

            DateTime completionDate = null;
            if (json.has("completionDate")) {
                completionDate = timeAdapter.unmarshal(json.getString("completionDate"));
            }

            String log = null;
            if (json.has("log")) {
                log = json.getString("log");
            }

            List<String> prevLogs = null;
            if (json.has("previousAttemptsLogs")) {
                prevLogs = new ArrayList<>();
                JSONArray jsonLogs = json.getJSONArray("previousAttemptsLogs");
                for (int i = 0; i < jsonLogs.length(); i++) {
                    prevLogs.add(jsonLogs.getString(i));
                }
            }

            return createStepState(state, failureCount, description, startDate, completionDate, log, prevLogs, skippable);

        } catch (JSONException e) {
            throw new IllegalArgumentException("Can't parse JSON:\n" + jsonString, e);
        }
    }

    protected abstract StepState createStepState(StepExecutionState state, int failureCount, String description,
                                                 DateTime startDate, DateTime completionDate, String log,
                                                 List<String> prevLogs, boolean skippable);
}
