/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization.util

import java.util

import com.xebialabs.deployit.plugin.api.services.Repository
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xltype.serialization.CiReference

import scala.jdk.CollectionConverters._

object ReferenceUtils {

  /**
    * Resolve all ciReferences by first looking up in ciMap and then querying repository
    *
    * @param references         - list of ci references
    * @param ciMap              - map of cis (context). May potentially be not persisted yet
    * @param repository         - ci repository instance
    * @param skipNotExistingCis - don't fail on not existing ci, but just keeping null instead.
    * @param referenceDepth     - how deep should we go while resolving references
    */
  def resolveReferences(references: util.List[CiReference],
                        ciMap: util.Map[String, ConfigurationItem],
                        repository: Repository,
                        skipNotExistingCis: Boolean,
                        referenceDepth: Integer): Unit = {
    val allIds = references.asScala.flatMap(_.getIds.asScala).toSet
    val ciContext = ciMap.asScala
    val idsToRead = allIds.filterNot(ciMap.containsKey)
    val loaded = if (idsToRead.nonEmpty) {
      repository.read[ConfigurationItem](idsToRead.toList.asJava, referenceDepth).asScala.map(ci => ci.getId -> ci).toMap
    } else Map.empty[String, ConfigurationItem]

    references.asScala.foreach { reference =>
      reference.set(reference.getIds.asScala.flatMap { id => ciContext.get(id).orElse(loaded.get(id)) }.asJava)
    }
  }

  /**
    * Resolve all ciReferences by first looking up in ciMap and then querying repository
    *
    * @param references - list of ci references
    * @param ciMap      - map of cis (context). May potentially be not persisted yet
    * @param repository - ci repository instance
    */
  def resolveReferences(references: util.List[CiReference],
                        ciMap: util.Map[String, ConfigurationItem],
                        repository: Repository): Unit =
    resolveReferences(references, ciMap, repository, skipNotExistingCis = false, Int.MaxValue)
}
