package com.xebialabs.xltype.serialization.xstream;

import java.util.List;
import com.thoughtworks.xstream.converters.Converter;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;

import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

import static com.xebialabs.xltype.serialization.xstream.Converters.readList;
import static com.xebialabs.xltype.serialization.xstream.Converters.writeConfigurationItem;


@XStreamProvider(tagName = "deployment", readable = Deployment.class)
public class DeploymentConverter implements Converter {
    @Override
    public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
        Deployment depl = (Deployment) source;
        if (depl.getId() != null) {
            writer.addAttribute("id", depl.getId());
        }
        writer.addAttribute("type", depl.getDeploymentType().name());
        writer.startNode("application");
        writeConfigurationItem(depl.getDeployedApplication(), writer, context);
        writer.endNode();
        writer.startNode("deployeds");
        for (ConfigurationItem configurationItem : depl.getDeployeds()) {
            writeConfigurationItem(configurationItem, writer, context);
        }
        writer.endNode();
        writeIds("deployables", depl.getDeployables(), writer, context);
        writeIds("containers", depl.getContainers(), writer, context);
    }

    private void writeIds(String nodeName, List<ConfigurationItemId> ids, HierarchicalStreamWriter writer, MarshallingContext context) {
        writer.startNode(nodeName);
        for (ConfigurationItemId configurationItemId : ids) {
            writerConfigurationItemId(configurationItemId, writer, context);
        }
        writer.endNode();
    }

    private void writerConfigurationItemId(ConfigurationItemId configurationItemId, HierarchicalStreamWriter writer, MarshallingContext context) {
        writer.startNode("ci");
        context.convertAnother(configurationItemId);
        writer.endNode();
    }

    @Override
    public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
        Deployment depl = new Deployment();
        depl.setId(reader.getAttribute("id"));
        depl.setDeploymentType(Deployment.DeploymentType.valueOf(reader.getAttribute("type")));
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            if ("application".equals(reader.getNodeName())) {
                reader.moveDown();
                ConfigurationItem ci = (ConfigurationItem) context.convertAnother(depl, ConfigurationItem.class);
                depl.setDeployedApplication(ci);
                reader.moveUp();
            } else if ("deployeds".equals(reader.getNodeName())) {
                depl.setDeployeds(readList(depl, ConfigurationItem.class, reader, context));
            } else if ("deployables".equals(reader.getNodeName())) {
                depl.setDeployables(readList(depl, ConfigurationItemId.class, reader, context));
            } else if ("containers".equals(reader.getNodeName())) {
                depl.setContainers(readList(depl, ConfigurationItemId.class, reader, context));
            }
            reader.moveUp();
        }
        return depl;
    }

    @Override
    public boolean canConvert(Class type) {
        return Deployment.class.isAssignableFrom(type);
    }
}
