package com.xebialabs.xltype.serialization.json;

import java.util.List;

import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;

import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.engine.api.dto.Deployment.DeploymentType;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xltype.serialization.CiWriter;
import com.xebialabs.xltype.serialization.ConfigurationItemConverter;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.collect.Lists.newArrayList;

public class DeploymentJsonConverter {

    private final ConfigurationItemConverter ciConverter;

    public DeploymentJsonConverter(ConfigurationItemConverter converter) {
        this.ciConverter = converter;
    }

    public ConfigurationItemConverter getCiConverter() {
        return ciConverter;
    }

    public String toJson(Deployment deployment) {
        JsonWriter writer = new JsonWriter();
        CiWriter ciWriter = new CiJsonWriter(writer);
        ConfigurationItemConverter converter = new ConfigurationItemConverter();

        writer.object();
        writer.key("type").value(deployment.getDeploymentType());

        writer.key("application");
        converter.writeCi(deployment.getDeployedApplication(), ciWriter);

        writer.key("deployeds");
        converter.writeCis(deployment.getDeployeds(), ciWriter);

        writer.key("deployables");
        ciWriter.typedCiReferences(deployment.getDeployables());

        writer.key("containers");
        ciWriter.typedCiReferences(deployment.getContainers());

        writer.endObject();

        return writer.toString();
    }

    public Deployment toDeployment(String jsonString) {
        try {
            JSONObject json = new JSONObject(jsonString);

            checkArgument(json.has("type"), "Missing 'type' property in JSON:\n%s", jsonString);
            checkArgument(json.has("application"), "Missing 'application' property in JSON:\n%s", jsonString);
            checkArgument(json.has("deployeds"), "Missing 'deployeds' property in JSON:\n%s", jsonString);

            Deployment deployment = new Deployment();

            DeploymentType type = DeploymentType.valueOf(json.getString("type"));
            deployment.setDeploymentType(type);

            ConfigurationItemConverter converter = new ConfigurationItemConverter();
            ConfigurationItem application = converter.readCi(new CiJsonReader(json.getJSONObject("application")));
            deployment.setDeployedApplication(application);

            List<ConfigurationItem> deployeds = converter.readCis(new CiListJsonReader(json.getJSONArray("deployeds")));
            deployment.setDeployeds(deployeds);

            deployment.setDeployables(toConfigurationItemIds(json.getJSONArray("deployables")));
            deployment.setContainers(toConfigurationItemIds(json.getJSONArray("containers")));

            return deployment;

        } catch (JSONException e) {
            throw new IllegalArgumentException("Can't parse JSON:\n" + jsonString, e);
        }
    }

    // TODO Move this to a more proper place
    private static List<ConfigurationItemId> toConfigurationItemIds(JSONArray jsonArray) throws JSONException {
        List<ConfigurationItemId> ids = newArrayList();
        for (int i = 0; i < jsonArray.length(); i++) {
            ids.add(toConfigurationItemId(jsonArray.getJSONObject(i)));
        }
        return ids;
    }

    private static ConfigurationItemId toConfigurationItemId(JSONObject jsonObject) throws JSONException {
        return new ConfigurationItemId(jsonObject.getString("ci"), com.xebialabs.deployit.plugin.api.reflect.Type.valueOf(jsonObject.getString("type")));
    }

}
