/**
 * Copyright 2014-2018 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization.json;

import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.engine.api.dto.Deployment.DeploymentType;
import com.xebialabs.deployit.plugin.api.deployment.ResolvedPlaceholder;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.xltype.serialization.CiWriter;
import com.xebialabs.xltype.serialization.ConfigurationItemConverter;
import org.codehaus.jettison.json.JSONArray;
import org.codehaus.jettison.json.JSONException;
import org.codehaus.jettison.json.JSONObject;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static com.xebialabs.xltype.serialization.json.Converters.checkArgument;

public class DeploymentJsonConverter {

    private final ConfigurationItemConverter ciConverter;

    public DeploymentJsonConverter(ConfigurationItemConverter converter) {
        this.ciConverter = converter;
    }

    public ConfigurationItemConverter getCiConverter() {
        return ciConverter;
    }

    public String toJson(Deployment deployment) {
        JsonWriter writer = new JsonWriter();
        toJson(deployment, writer);
        return writer.toString();
    }

    public void toJson(final Deployment deployment, final JsonWriter writer) {
        CiWriter ciWriter = new CiJsonWriter(writer);

        writer.object();

        if (deployment.getId() != null) {
            writer.key("id").value(deployment.getId());
        }

        writer.key("type").value(deployment.getDeploymentType());

        writer.key("deploymentGroupIndex").value(deployment.getDeploymentGroupIndex());

        writer.key("application");
        ciConverter.writeCi(deployment.getDeployedApplication(), ciWriter);

        writer.key("deployeds");
        ciConverter.writeCis(deployment.getDeployeds(), ciWriter);

        writer.key("deployables");
        ciWriter.typedCiReferences(deployment.getDeployables());

        writer.key("containers");
        ciWriter.typedCiReferences(deployment.getContainers());

        writer.key("requiredDeployments");
        ciWriter.startList();
        deployment.getRequiredDeployments().forEach(requiredDeployment -> toJson(requiredDeployment, writer));
        ciWriter.endList();

        toJson(deployment.getResolvedPlaceholders(), writer);

        writer.endObject();
    }

    private List<Deployment> toDeployments(JSONArray jsonArray) throws JSONException {
        List<Deployment> deployments = new ArrayList<>();
        for (int i = 0; i < jsonArray.length(); i++) {
            deployments.add(toDeployment(jsonArray.getJSONObject(i)));
        }
        return deployments;
    }

    private Set<ResolvedPlaceholder> toResolvedPlaceholders(JSONArray jsonArray) throws JSONException {
        Set<ResolvedPlaceholder> placeholders = new HashSet<>();
        for (int i = 0; i < jsonArray.length(); i++) {
            placeholders.add(toResolvedPlaceholder(jsonArray.getJSONObject(i)));
        }
        return placeholders;
    }

    public void toJson(final Set<ResolvedPlaceholder> resolvedPlaceholders, final JsonWriter json) {
        json.key("resolvedPlaceholders");
        json.array();
        for (ResolvedPlaceholder resolvedPlaceholder : resolvedPlaceholders) {
            json.object();
            json.key("encrypted").value(resolvedPlaceholder.isEncrypted());
            json.key("containerId").value(resolvedPlaceholder.containerId());
            json.key("containerDeleted").value(resolvedPlaceholder.containerDeleted());
            json.key("deployedAppId").value(resolvedPlaceholder.deployedAppId());
            json.key("dictionaryId").value(resolvedPlaceholder.dictionaryId());
            json.key("dictionaryDeleted").value(resolvedPlaceholder.dictionaryDeleted());
            json.key("environmentId").value(resolvedPlaceholder.environmentId());
            json.key("environmentDeleted").value(resolvedPlaceholder.environmentDeleted());
            json.key("versionId").value(resolvedPlaceholder.versionId());
            json.key("key").value(resolvedPlaceholder.key());
            json.key("value").value(resolvedPlaceholder.value());
            json.endObject();
        }
        json.endArray();
    }

    public Deployment toDeployment(String jsonString) {
        try {
            JSONObject json = new JSONObject(jsonString);
            return toDeployment(json);
        } catch (JSONException e) {
            throw new IllegalArgumentException("Can't parse JSON:\n" + jsonString, e);
        }
    }

    public ResolvedPlaceholder toResolvedPlaceholder(final JSONObject jsonObject) throws JSONException {
        return new ResolvedPlaceholder(
                jsonObject.getBoolean("encrypted"),
                jsonObject.getString("key"),
                jsonObject.getString("value"),
                jsonObject.getString("containerId"),
                jsonObject.getBoolean("containerDeleted"),
                jsonObject.getString("deployedAppId"),
                jsonObject.getString("dictionaryId"),
                jsonObject.getBoolean("dictionaryDeleted"),
                jsonObject.getString("environmentId"),
                jsonObject.getBoolean("environmentDeleted"),
                jsonObject.getString("versionId")
        );
    }

    public Deployment toDeployment(final JSONObject json) throws JSONException {
        final String jsonString = json.toString();
        checkArgument(json.has("type"), "Missing 'type' property in JSON:\n%s", jsonString);
        checkArgument(json.has("application"), "Missing 'application' property in JSON:\n%s", jsonString);
        checkArgument(json.has("deployeds"), "Missing 'deployeds' property in JSON:\n%s", jsonString);

        Deployment deployment = new Deployment();

        if (json.has("id")) {
            deployment.setId(json.getString("id"));
        }

        DeploymentType type = DeploymentType.valueOf(json.getString("type"));
        deployment.setDeploymentType(type);

        if (!json.getString("deploymentGroupIndex").isEmpty()) {
            deployment.setDeploymentGroupIndex(Integer.valueOf(json.getString("deploymentGroupIndex")));
        }

        ConfigurationItem application = ciConverter.readCi(new CiJsonReader(json.getJSONObject("application")));
        deployment.setDeployedApplication(application);

        List<ConfigurationItem> deployeds = ciConverter.readCis(new CiListJsonReader(json.getJSONArray("deployeds")));
        deployment.setDeployeds(deployeds);

        deployment.setDeployables(toConfigurationItemIds(json.getJSONArray("deployables")));
        deployment.setContainers(toConfigurationItemIds(json.getJSONArray("containers")));
        deployment.setRequiredDeployments(toDeployments(json.getJSONArray("requiredDeployments")));

        if (json.has("resolvedPlaceholders")) {
            deployment.addResolvedPlaceholders(toResolvedPlaceholders(json.getJSONArray("resolvedPlaceholders")));
        }

        return deployment;
    }

    // TODO Move this to a more proper place
    private static List<ConfigurationItemId> toConfigurationItemIds(JSONArray jsonArray) throws JSONException {
        List<ConfigurationItemId> ids = new ArrayList<>();
        for (int i = 0; i < jsonArray.length(); i++) {
            ids.add(toConfigurationItemId(jsonArray.getJSONObject(i)));
        }
        return ids;
    }

    private static ConfigurationItemId toConfigurationItemId(JSONObject jsonObject) throws JSONException {
        return new ConfigurationItemId(jsonObject.getString("ci"), com.xebialabs.deployit.plugin.api.reflect.Type.valueOf(jsonObject.getString("type")));
    }

}
