/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.xltype.serialization.xstream;

import java.util.*;

import com.xebialabs.deployit.plugin.api.udm.ExternalProperty;
import com.xebialabs.deployit.plugin.api.udm.lookup.LookupValueKey;
import org.joda.time.DateTime;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;

import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.xltype.serialization.CiListReader;
import com.xebialabs.xltype.serialization.CiReader;

public class CiXstreamReader implements CiReader, CiListReader {
    private final HierarchicalStreamReader reader;
    private final DateTimeAdapter dateTimeAdapter = new DateTimeAdapter();

    public CiXstreamReader(HierarchicalStreamReader reader) {
        this.reader = reader;
    }

    @Override
    public String getType() {
        return reader.getNodeName();
    }

    @Override
    public String getId() {
        return reader.getAttribute("id");
    }

    @Override
    public String getToken() {
        return reader.getAttribute("token");
    }

    @Override
    public CiAttributes getCiAttributes() {
        return new CiAttributes(
                reader.getAttribute("created-by"),
                getDateAttribute("created-at"),
                reader.getAttribute("last-modified-by"),
                getDateAttribute("last-modified-at"),
                getIntegerAttribute("scm-traceability-data-id")
        );
    }

    private DateTime getDateAttribute(String attrName) {
        final String attribute = reader.getAttribute(attrName);
        if (attribute != null) {
            return dateTimeAdapter.unmarshal(attribute);
        }
        return null;
    }

    private Integer getIntegerAttribute(String attrName) {
        final String attribute = reader.getAttribute(attrName);
        if (attribute != null) {
            return Integer.valueOf(attribute);
        }
        return null;
    }

    @Override
    public boolean hasMoreProperties() {
        return reader.hasMoreChildren();
    }

    @Override
    public void moveIntoProperty() {
        reader.moveDown();
    }

    @Override
    public CiReader moveIntoNestedProperty() {
        reader.moveDown();
        return this;
    }

    @Override
    public void moveOutOfProperty() {
        reader.moveUp();
    }

    @Override
    public String getCurrentPropertyName() {
        return reader.getNodeName();
    }

    @Override
    public String getStringValue() {
        return reader.getValue();
    }

    @Override
    public List<String> getStringValues() {
        List<String> strings = new ArrayList<>();

        while (reader.hasMoreChildren()) {
            reader.moveDown();
            strings.add(reader.getValue());
            reader.moveUp();
        }

        return strings;
    }

    @Override
    public Map<String, String> getStringMap() {
        Map<String, String> map = new LinkedHashMap<>();

        while (reader.hasMoreChildren()) {
            reader.moveDown();
            map.put(reader.getAttribute("key"), reader.getValue());
            reader.moveUp();
        }

        return map;
    }

    @Override
    public boolean isCiReference() {
        return reader.getAttribute("ref") != null;
    }

    @Override
    public String getCiReference() {
        return reader.getAttribute("ref");
    }

    @Override
    public List<String> getCiReferences() {
        List<String> references = new ArrayList<>();

        while (reader.hasMoreChildren()) {
            reader.moveDown();
            references.add(getCiReference());
            reader.moveUp();
        }

        return references;
    }

    @Override
    public CiListReader getCurrentCiListReader() {
        return this;
    }

    @Override
    public List<ValidationMessage> getValidationMessages() {
        List<ValidationMessage> messages = new ArrayList<>();
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            String property = reader.getAttribute("property");
            String level = reader.getAttribute("level");
            messages.add(new ValidationMessage(reader.getAttribute("ci"), property, getStringValue(), level));
            reader.moveUp();
        }

        return messages;
    }

    @Override
    public Map<String, ExternalProperty> getExternalProperties() {
        HashMap<String, ExternalProperty> result = new HashMap<>();
        while (reader.hasMoreChildren()) {
            reader.moveDown();
            String propertyName = reader.getNodeName();
            String kind = reader.getAttribute("kind");
            if (kind.equals("lookup")) {
                LookupValueKey lookupValueKey = new LookupValueKey();
                lookupValueKey.setKey(reader.getAttribute("key"));
                lookupValueKey.setProviderId(reader.getAttribute("provider"));
                result.put(propertyName, lookupValueKey);
            } else {
                throw new IllegalArgumentException("Unknown external property kind: '" + kind +"'.");
            }
            reader.moveUp();
        }
        return result;
    }

    @Override
    public boolean hasMoreChildren() {
        return reader.hasMoreChildren();
    }

    @Override
    public void moveIntoChild() {
        reader.moveDown();
    }

    @Override
    public void moveOutOfChild() {
        reader.moveUp();
    }

    @Override
    public CiReader getCurrentCiReader() {
        return this;
    }
}
