package com.xebialabs.json_patch

import java.io.InputStream

import com.fasterxml.jackson.annotation.{JsonCreator, JsonProperty}
import com.fasterxml.jackson.core.JsonPointer
import com.fasterxml.jackson.databind.{JsonNode, ObjectMapper}
import JsonActivators._

object JsonActivators {
  case class ActivatorConditionInput(path: String, value: Any, node: Any)
  case class ActivatorCondition(pathOnly: Boolean,
                                check: ActivatorConditionInput => Boolean,
                                formatExplanation: ActivatorConditionInput => String)

  val conditions: Map[String, ActivatorCondition] = Map(
    "equals" -> ActivatorCondition(
      pathOnly = false,
      input => input.node.equals(input.value),
      input => s"""Value ${input.value.toString} is not equal to ${input.node.toString}."""
    ),
    "exists" -> ActivatorCondition(
      pathOnly = true,
      input => !input.node.asInstanceOf[JsonNode].isMissingNode,
      input => s"""Path "${input.path}" not found."""
    )
  )
}

class JsonActivators extends JsonSupport {
  def checkActivatorsJson(documentStream: InputStream, activators: String): Boolean =
    checkActivators(jsonMapper, jsonMapper.readTree(documentStream), activators)

  def checkActivatorsYaml(documentStream: InputStream, activators: String): Boolean =
    checkActivators(yamlMapper, yamlMapper.readTree(documentStream), activators)

  def checkActivators(valueMapper: ObjectMapper, documentNode: JsonNode, activators: String): Boolean = {
    if (isNullOrEmpty(activators)) {
      return true
    }

    val activatorsNodes = jsonMapper.readTree(activators)
    val transformedActivators = Utils.transformValues(valueMapper, activatorsNodes)
    val listOfActivators = jsonMapper.convertValue(transformedActivators, classOf[Array[JsonActivator]]).toList
    validateActivators(listOfActivators)

    listOfActivators.forall { activator =>
      val pointer = JsonPointer.valueOf(activator.path)
      val jsonNode: JsonNode = documentNode.at(pointer)
      conditions.get(activator.cond) match {
        case Some(condition) => condition.check(ActivatorConditionInput(activator.path, activator.value, jsonNode))
        case None => throw new Exception(s"Unknown operation: ${activator.cond}")
      }
    }
  }

  private def validateActivators(listOfActivators: List[JsonActivator]): Unit = listOfActivators.foreach { activator =>
    if (isNullOrEmpty(activator.path)) {
      throw new IllegalArgumentException(s"'path' field required for activator: ${activator}")
    }
    if (isNullOrEmpty(activator.cond)) {
      throw new IllegalArgumentException(s"'op' field required for activator: ${activator}")
    }
    if (activator.cond.equals("equals") && activator.value.isNull) {
      throw new IllegalArgumentException(s"'value' field required for activator when 'op' is equals: ${activator}")
    }
  }

  private def isNullOrEmpty(s: String) = s == null || s.trim.isEmpty
}

case class JsonActivator @JsonCreator()(@JsonProperty("path") path: String, @JsonProperty("cond") cond: String, @JsonProperty("value") value: JsonNode)
