package com.xebialabs.deployit.booter.local;

import com.xebialabs.deployit.booter.local.utils.ReflectionUtils;
import com.xebialabs.deployit.plugin.api.reflect.*;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.plugin.api.validation.Validator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import java.lang.reflect.Field;
import java.util.*;

import static com.xebialabs.deployit.booter.local.utils.ReflectionUtils.searchField;
import static com.xebialabs.deployit.booter.local.utils.Strings.deCamelize;
import static com.xebialabs.deployit.booter.local.utils.Strings.isBlank;


public abstract class LocalDescriptor implements Descriptor {

    protected final Logger logger = LoggerFactory.getLogger(getClass());

    private Type type;
    private Class<? extends ConfigurationItem> clazz;
    private String label;
    private String description;
    private Optional<String> rootName = Optional.empty();
    private final List<Type> superclasses = new ArrayList<>();
    private final Set<Type> interfaces = new HashSet<>();
    private boolean virtual = false;
    private boolean versioned = true;
    private String icon;
    private boolean isInspectable;
    private Map<String, LocalPropertyDescriptor> properties = new LinkedHashMap<>();
    private Map<String, MethodDescriptor> controlTasks = new HashMap<>();
    private Type deployableType;
    private Type containerType;
    protected List<Validator<ConfigurationItem>> validators = new ArrayList<>();
    protected transient List<TypeVerification> verifications = new ArrayList<>();

    private Field syntheticPropertiesField;

    @Override
    public Type getType() {
        return type;
    }

    protected void setType(Type type) {
        this.type = type;
    }

    @Override
    public Class<? extends ConfigurationItem> getClazz() {
        return clazz;
    }

    protected void setClazz(Class<? extends ConfigurationItem> clazz) {
        this.clazz = clazz;
    }

    @Override
    public String getRootName() {
        return rootName.orElse("");
    }

    protected Optional<String> getRootNameMaybe() {
        return rootName;
    }

    protected void setRootName(Optional<String> rootName) {
        this.rootName = rootName.map(this::firstCharToUpperCase);
    }

    private String firstCharToUpperCase(String value) {
        return value.substring(0, 1).toUpperCase() + value.substring(1, value.length()).toLowerCase();
    }

    @Override
    public String getLabel() {
        return label;
    }

    protected void setLabel(String label) {
        this.label = label;
    }

    @Override
    public List<Type> getSuperClasses() {
        return Collections.unmodifiableList(superclasses);
    }

    protected void addSuperClass(Type supertype) {
        superclasses.add(supertype);
    }

    @Override
    public Set<Type> getInterfaces() {
        return Collections.unmodifiableSet(interfaces);
    }

    protected void addInterface(Type intf) {
        interfaces.add(intf);
    }

    @Override
    public String getIcon() {
        return icon;
    }

    protected void setIcon(String icon) {
        this.icon = icon;
    }

    @Override
    public boolean isVersioned() {
        return versioned;
    }

    protected void setVersioned(boolean versioned) {
        this.versioned = versioned;
    }

    @Override
    public boolean isVirtual() {
        return virtual;
    }

    protected void setVirtual(boolean virtual) {
        this.virtual = virtual;
    }

    @Override
    public String getDescription() {
        return description;
    }

    protected void setDescription(String description) {
        this.description = description;
    }

    @Override
    public boolean isInspectable() {
        return isInspectable;
    }

    protected void setInspectable(boolean inspectable) {
        isInspectable = inspectable;
    }

    @Override
    public boolean isAssignableTo(Class<?> clazz) {
        return isAssignableTo(Type.valueOf(clazz));
    }

    @Override
    public boolean isAssignableTo(Type type) {
        return this.getType().isSubTypeOf(type) || this.getType().equals(type);
    }

    protected void addPropertyDescriptor(LocalPropertyDescriptor propertyDescriptor) {
        properties.put(propertyDescriptor.getName(), propertyDescriptor);
    }

    @Override
    public Collection<PropertyDescriptor> getPropertyDescriptors() {
        return new ArrayList<>(properties.values());
    }

    protected Map<String, LocalPropertyDescriptor> getPropertyDescriptorsAsMap() {
        return Collections.unmodifiableMap(properties);
    }

    @Override
    public PropertyDescriptor getPropertyDescriptor(String name) {
        return properties.get(name);
    }

    protected void addControlTask(MethodDescriptor from) {
        controlTasks.put(from.getName(), from);
    }

    @Override
    public MethodDescriptor getControlTask(String name) {
        return controlTasks.get(name);
    }

    @Override
    public Collection<MethodDescriptor> getControlTasks() {
        return Collections.unmodifiableCollection(controlTasks.values());
    }

    protected Map<String, MethodDescriptor> getControlTasksAsMap() {
        return Collections.unmodifiableMap(controlTasks);
    }

    protected void applyTypeModification(Element modificationElement) {
        SyntheticParser parser = SyntheticParser.PARSER;
        setVersioned(parser.parseVersioned(modificationElement).orElse(isVersioned()));
        validators.addAll(parser.parseValidators(modificationElement, getType()));
        verifications.addAll(parser.parseVerifications(modificationElement, getType()));
        parser.parseControlTasks(this, modificationElement).forEach(this::validateAndAddControlTask);
        parser.parseProperties(this, modificationElement).forEach(this::overrideOrAddPropertyDescriptor);
    }

    protected void validateAndAddControlTask(MethodDescriptor methodDescriptor) {
        verifyNewControlTask(methodDescriptor);
        addControlTask(methodDescriptor);
    }

    protected void verifyNewControlTask(MethodDescriptor controlTask) {
        if (getControlTasksAsMap().containsKey(controlTask.getName())) {
            throw new IllegalStateException("Cannot override existing Control Task [" + controlTask.getFqn() + "] with a synthetic one.");
        }
    }

    protected void overrideOrAddPropertyDescriptor(LocalPropertyDescriptor propertyDescriptor) {
        LocalPropertyDescriptor existingProperty = getPropertyDescriptorsAsMap().get(propertyDescriptor.getName());
        if(existingProperty != null) {
            addPropertyDescriptor(new ExtendByPropertyDescriptor(propertyDescriptor, existingProperty));
        } else {
            addPropertyDescriptor(propertyDescriptor);
        }
    }

    private void initSyntheticPropertiesField() {
        if (!getClazz().isInterface() && !Type.valueOf("api.ValidatedConfigurationItem").equals(getType())) {
            syntheticPropertiesField = searchField(getClazz(), ConfigurationItem.SYNTHETIC_PROPERTIES_FIELD);
        }
    }

    protected Object getSyntheticPropertyValue(ConfigurationItem configurationItem, String propertyName) {
        Map<String, Object> synth = getSyntheticPropertyMap(configurationItem);
        return synth.get(propertyName);
    }

    protected void setSyntheticPropertyValue(ConfigurationItem configurationItem, String propertyName, Object value) {
        getSyntheticPropertyMap(configurationItem).put(propertyName, value);
    }

    @Override
    public Type getDeployableType() {
        return deployableType;
    }

    protected void setContainerType(Type containerType) {
        this.containerType = containerType;
    }

    @Override
    public Type getContainerType() {
        return containerType;
    }

    protected void setDeployableType(Type deployableType) {
        this.deployableType = deployableType;
    }

    @SuppressWarnings("unchecked")
    private Map<String, Object> getSyntheticPropertyMap(ConfigurationItem configurationItem) {
        return (Map<String, Object>) ReflectionUtils.getField(configurationItem, syntheticPropertiesField);
    }

    void verify(final Verifications verifications) {
        DescriptorVerification.verify(verifications, this);
        for (TypeVerification verification : this.verifications) {
            verification.verify(this, (message, params) -> verifications.verify(getType(), false, message, params));
        }
        verifySyntheticPropertiesField(verifications);
    }

    private void verifySyntheticPropertiesField(Verifications verifications) {
        verifications.verify(getType(), syntheticPropertiesField != null, "Synthetic properties field should be set");
        if (syntheticPropertiesField != null) {
            verifications.verify(getType(), syntheticPropertiesField.getType().isAssignableFrom(Map.class),
                    "Synthetic properties field should be Map<String, Object>, not: %s", syntheticPropertiesField.getType());
        }
    }


    @Override
    public <T extends ConfigurationItem> T newInstance(String id) {
        if (isVirtual()) {
            throw new IllegalArgumentException("Cannot instantiate class for " + getType() + " because it is virtual");
        }
        Class<?> clazz = getClazz();

        try {
            Field typeField = searchField(clazz, ConfigurationItem.TYPE_FIELD);
            //noinspection unchecked
            T t = (T) clazz.newInstance();
            t.setId(id);
            typeField.set(t, getType());

            prefillDefaultProperties(t);
            callPostProcessor(t);
            return t;
        } catch (InstantiationException | IllegalAccessException exc) {
            throw new RuntimeException("Cannot instantiate class " + clazz.getName(), exc);
        }
    }

    private <T extends ConfigurationItem> void prefillDefaultProperties(T t) {
        for (PropertyDescriptor pd : getPropertyDescriptors()) {
            LocalPropertyDescriptor lpd = (LocalPropertyDescriptor) pd;
            if (pd.getDefaultValue() != null) {
                pd.set(t, lpd.getDefaultValue());
            } else {
                pd.set(t, lpd.emptyValue());
            }
        }
    }

    private void callPostProcessor(ConfigurationItem ci) {
        ConfigurationItemPostProcessors.getProcessors(getType()).stream().forEach(processor -> processor.process(ci));
    }

    @Override
    public boolean areEqual(ConfigurationItem item, ConfigurationItem other) {
        return areEqualDeeply(item, other, new HashSet<>());
    }

    protected boolean areEqualDeeply(ConfigurationItem item, ConfigurationItem other, Set<String> itemsCompared) {
        if (item == null) {
            return other == null;
        }

        if (!getType().equals(item.getType()) || !getType().equals(other.getType())) {
            return false;
        }

        if (!itemsCompared.add(item.getId())) {
            return true;
        }

        for (PropertyDescriptor pd : getPropertyDescriptors()) {
            if (!((LocalPropertyDescriptor) pd).areEqual(item, other, itemsCompared)) {
                return false;
            }
        }
        return true;
    }

    protected String toLabel(Type type) {
        return deCamelize(type.getPrefix()) + ": " + deCamelize(type.getName());
    }

    protected void initHierarchy() {
        if (!getSuperClasses().isEmpty()) {
            Type toInitFrom = getSuperClasses().get(0);

            LocalDescriptor superDesc = (LocalDescriptor) LocalDescriptorRegistry.getDescriptor(toInitFrom);
            if (superDesc == null) {
                throw new IllegalStateException("Cannot build type hierarchy for " + getType() + " because one of its supertypes cannot be found: " + toInitFrom + " not found");
            }

            // For synthetic types, root and clazz might not be set yet.
            if (!getRootNameMaybe().isPresent()) {
                setRootName(superDesc.getRootNameMaybe());
            }

            if (getClazz() == null) {
                setClazz(superDesc.getClazz());
            }

            if (getLabel() == null) {
                setLabel(toLabel(getType()));
            }

            if (isBlank(getIcon())) {
                setIcon(superDesc.getIcon());
            }

            inheritPropertyDescriptors(superDesc);
            inheritControlTasks(superDesc);
            inheritValidators(validators, superDesc.validators);
            inheritVerifications(verifications, superDesc.verifications);

            superDesc.getSuperClasses().forEach(this::addSuperClass);
            superDesc.getInterfaces().forEach(this::addInterface);

            if (getDeployableType() == null) {
                setDeployableType(superDesc.getDeployableType());
            }

            if (getContainerType() == null) {
                setContainerType(superDesc.getContainerType());
            }
        }
        initSyntheticPropertiesField();
    }

    private void inheritValidators(List<Validator<ConfigurationItem>> dest, List<Validator<ConfigurationItem>> source) {
        dest.addAll(source);
    }

    private void inheritVerifications(List<TypeVerification> dest, List<TypeVerification> src) {
        dest.addAll(src);
    }

    private void inheritControlTasks(LocalDescriptor superTypeDescriptor) {
        for (Map.Entry<String, MethodDescriptor> sourceEntry : superTypeDescriptor.getControlTasksAsMap().entrySet()) {
            if (!getControlTasksAsMap().containsKey(sourceEntry.getKey())) {
                addControlTask(new LocalMethodDescriptor((LocalMethodDescriptor) sourceEntry.getValue(), this));
            } else {
                logger.warn("Not inheriting ControlTask [{}] on [{}]", new Object[]{sourceEntry.getValue().getFqn(), getType()});
            }
        }
    }

    private void inheritPropertyDescriptors(LocalDescriptor superTypeDescriptor) {
        Map<String, LocalPropertyDescriptor> originalProperties = new LinkedHashMap<>(properties);
        properties.clear();
        for (Map.Entry<String, LocalPropertyDescriptor> sourceEntry : superTypeDescriptor.getPropertyDescriptorsAsMap().entrySet()) {
            if (!originalProperties.containsKey(sourceEntry.getKey())) {
                properties.put(sourceEntry.getKey(), sourceEntry.getValue().copyWithNewDescriptor(this));
            } else {
                LocalPropertyDescriptor originalPropertyDescriptor = originalProperties.get(sourceEntry.getKey());
                originalProperties.put(sourceEntry.getKey(), new ExtendByPropertyDescriptor(originalPropertyDescriptor, sourceEntry.getValue()));
            }
        }
        properties.putAll(originalProperties);
    }

    @Override
    public List<ValidationMessage> validate(final ConfigurationItem ci) {
        final ArrayList<ValidationMessage> messages = new ArrayList<>();
        for (PropertyDescriptor propertyDescriptor : getPropertyDescriptors()) {
            ((LocalPropertyDescriptor) propertyDescriptor).validate(ci, messages);
        }
        for (Validator<ConfigurationItem> validator : validators) {
            validator.validate(ci, (message, params) -> messages.add(new ValidationMessage(ci.getId(), null, String.format(message, params))));
        }
        return messages;
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + "[" + getType() + "]";
    }
}
