package com.xebialabs.deployit.booter.local;

import java.io.*;

final class LocalGlobalContextManager extends AbstractGlobalContextManager {

    public void loadStoredDefaults(File configFile) {
        if (!configFile.exists()) {
            logger.info("Could not find '" + configFile + "', continuing without loading defaults");
            return;
        } else if (configFile.isDirectory()) {
            logger.error(configFile + " is a directory but should be a plain file, aborting!");
            throw new IllegalStateException("Please remove " + configFile + " and try again.");
        }
        try (BufferedReader br = new BufferedReader(new FileReader(configFile))){
            br.lines().forEach(this::processLine);
        } catch (IOException e) {
            throw new IllegalArgumentException("Could not read '" + configFile.toString() + "'", e);
        }
    }

    private void processLine(String line) {
        if (!line.startsWith("#") && line.contains("=")) {
            int i = line.indexOf("=");
            String key = line.substring(0, i);
            addContextEntry(key, line.substring(i + 1));
        }
    }

    public void storeDefaults(File configFile) {
        if (configFile != null) {

            if (!configFile.getParentFile().exists() || !configFile.getParentFile().isDirectory()) {
                logger.warn("Not writing {} because the directory does not exist", configFile);
                return;
            }
            try (BufferedWriter bufferedWriter = new BufferedWriter(new FileWriter(configFile))) {
                bufferedWriter.append(header());
                bufferedWriter.newLine();
                bufferedWriter.newLine();

                getContextEntries().forEach(pair -> {
                    GlobalContext.Value v = pair.value;
                    if (v.isShouldWrite()) {
                        try {
                            if (v.description != null && !v.description.isEmpty()) {
                                bufferedWriter.append("# ").append(toComment(v));
                                bufferedWriter.newLine();
                            }
                            bufferedWriter.append(v.isExplicit() ? "" : "#")
                                    .append(pair.key).append('=').append(v.getValue());
                            bufferedWriter.newLine();
                        } catch (IOException e) {
                            logger.error("Could not write to '" + configFile + "'", e);
                        }
                    }
                });
            } catch (FileNotFoundException e) {
                logger.error("Could not start writing to '" + configFile + "'", e);
            } catch (IOException e) {
                logger.error("Could not write to '" + configFile + "'", e);
            }
        }
    }
}
