/**
 * Copyright © 2014-2016 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */
package com.xebialabs.deployit.packager;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Version;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.util.OverthereUtils;
import de.schlichtherle.truezip.file.TArchiveDetector;
import de.schlichtherle.truezip.file.TFile;
import de.schlichtherle.truezip.file.TFileOutputStream;
import de.schlichtherle.truezip.file.TVFS;
import de.schlichtherle.truezip.fs.FsSyncException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collection;

import static com.xebialabs.deployit.packager.CiNameCache.id;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;
import static java.lang.String.format;
import static java.util.EnumSet.of;

public class DarPackager {

    private final CiNameCache cache;
    private ManifestWriterDriver manifestDriver;

    public DarPackager(ManifestWriter writer) {
        cache = new CiNameCache();
        manifestDriver = new ManifestWriterDriver(cache, writer);
    }

    public TFile buildPackage(Version version, String targetDir) {
        return buildPackage(version, targetDir, false);
    }

    public TFile buildPackage(Version version, String targetDir, boolean writeFullPaths) {
        String darName = darName(version);
        TFile darPackage = new TFile(targetDir, darName);
        try {
            manifestDriver.writeToPackage(version, darPackage, writeFullPaths);
            writeArtifactsToPackage(version, darPackage);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } finally {
            try {
                TVFS.umount(darPackage);
            } catch (FsSyncException e) {
                logger.warn("Couldn't cleanly umount the dar package {}", darPackage);
            }
        }
        return darPackage;
    }

    @SuppressWarnings("unchecked")
    private void writeArtifactsToPackage(ConfigurationItem ci, TFile darPackage) {
        if (manifestDriver.isLocalArtifact(ci)) {
            if (ci instanceof FolderArtifact) {
                OverthereFile sourceFile = ((Artifact) ci).getFile();
                logger.debug("Writing {} to package {}", sourceFile, darPackage);
                TFile artifactDir = new TFile(darPackage, cache.lookup(ci), TArchiveDetector.NULL);
                artifactDir.mkdirs();
                if (sourceFile.isFile()) {
                    copyFile(sourceFile, new TFile(artifactDir, sourceFile.getName(), TArchiveDetector.NULL));
                } else {
                    TFile targetFolder = new TFile(artifactDir, sourceFile.getName());
                    targetFolder.mkdirs();
                    copyFolder(sourceFile, targetFolder);
                }
            } else if (ci instanceof Artifact) {
                OverthereFile sourceFile = ((Artifact) ci).getFile();
                logger.debug("Writing {} to package {}", sourceFile, darPackage);
                TFile artifactDir = new TFile(darPackage, cache.lookup(ci), TArchiveDetector.NULL);
                artifactDir.mkdirs();
                TFile targetFile = new TFile(artifactDir, sourceFile.getName(), TArchiveDetector.NULL);
                copyFile(sourceFile, targetFile);
            }
        }
        for (PropertyDescriptor pd : ci.getType().getDescriptor().getPropertyDescriptors()) {
            if (pd.isAsContainment() && of(LIST_OF_CI, SET_OF_CI).contains(pd.getKind())) {
                Collection<ConfigurationItem> cis = (Collection<ConfigurationItem>) pd.get(ci);
                if (cis == null) {
                    logger.warn("{}: {}.{} (as-containment, kind {}) should not be null", ci.getId(), ci.getType(), pd.getName(), pd.getKind());
                    continue;
                }
                for (ConfigurationItem c : cis) {
                    writeArtifactsToPackage(c, darPackage);
                }
            }
        }
    }

    private void copyFolder(OverthereFile sourceFile, TFile targetFolder) {
        for (OverthereFile overthereFile : sourceFile.listFiles()) {
            if (overthereFile.isFile()) {
                copyFile(overthereFile, new TFile(targetFolder, overthereFile.getName(), TArchiveDetector.NULL));
            } else if (overthereFile.isDirectory()) {
                TFile targetFolder1 = new TFile(targetFolder, overthereFile.getName());
                targetFolder1.mkdirs();
                copyFolder(overthereFile, targetFolder1);
            }
        }
    }

    private void copyFile(final OverthereFile sourceFile, TFile targetFile) {
        try (InputStream is = sourceFile.getInputStream(); OutputStream os = new TFileOutputStream(targetFile)) {
            OverthereUtils.write(is, os);
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        }
    }

    private String darName(Version version) {
        return format("%s-%s.dar", version.getApplication().getName(), version.getVersion());
    }

    private static final Logger logger = LoggerFactory.getLogger(DarPackager.class);
}
