package com.xebialabs.deployit.packager;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.*;
import javax.xml.bind.DatatypeConverter;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.ImmutableList;
import com.google.common.io.CharStreams;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Version;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;

import de.schlichtherle.truezip.file.TFile;

import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.packager.truezip.TFiles.newOutputSupplier;
import static com.xebialabs.deployit.packager.CiNameCache.id;

public class ManifestWriterDriver {

    public static final HashSet<String> APPLICATION_PROPERTIES_TO_SKIP = newHashSet("application");

    public static final Set<String> ARTIFACT_PROPERTIES_TO_SKIP = newHashSet("placeholders");

    private final CiNameCache cache;
    private ManifestWriter writer;

    public ManifestWriterDriver(CiNameCache cache, ManifestWriter writer) {
        this.writer = writer;
        this.cache = cache;
    }

    public void writeToPackage(Version version, TFile file) throws IOException {
        TFile manifestFile = new TFile(file, writer.getManifestFileName());
        write(version);
        CharStreams.write(writer.toString(), CharStreams.newWriterSupplier(newOutputSupplier(manifestFile), Charset.defaultCharset()));
    }

    @VisibleForTesting
    void write(Version version) {
        cache.scan(version);
        writeVersion(version);
        writer.endManifest();
    }

    private void writeProperties(ConfigurationItem ci, ManifestWriter.ManifestCiWriter ciWriter) {
        for (PropertyDescriptor propertyDescriptor : ci.getType().getDescriptor().getPropertyDescriptors()) {
            writeProperty(propertyDescriptor, ci, ciWriter);
        }
    }

    private ManifestWriter.ManifestCiWriter writeVersion(Version version) {
        ManifestWriter.ManifestCiWriter manifestCiWriter = writer.writeVersion(version.getType(), cache.lookup(version), cache.lookup(version.getApplication()));
        for (PropertyDescriptor pd : version.getType().getDescriptor().getPropertyDescriptors()) {
            if (!APPLICATION_PROPERTIES_TO_SKIP.contains(pd.getName())) {
                writeProperty(pd, version, manifestCiWriter);
            }
        }
        return manifestCiWriter;
    }

    @SuppressWarnings("unchecked")
    private void writeProperty(PropertyDescriptor pd, ConfigurationItem ci, ManifestWriter.ManifestCiWriter manifestCiWriter) {
        Object o = pd.get(ci);

        if (o == null) return;
        if (pd.isHidden()) return;
        if (ci.getType().instanceOf(Type.valueOf(Artifact.class)) && ARTIFACT_PROPERTIES_TO_SKIP.contains(pd.getName())) return;

        manifestCiWriter.property(pd.getName());
        switch (pd.getKind()) {
            case BOOLEAN:
            case INTEGER:
            case STRING:
                manifestCiWriter.writeAsStringValue(o.toString());
                break;
            case ENUM:
                manifestCiWriter.writeAsStringValue(((Enum<?>) pd.get(ci)).name());
                break;
            case DATE:
                Calendar c = Calendar.getInstance();
                c.setTime((Date) o);
                manifestCiWriter.writeAsStringValue(DatatypeConverter.printDateTime(c));
                break;
            case SET_OF_STRING:
            case LIST_OF_STRING:
                manifestCiWriter.writeStringCollectionProperty(ImmutableList.copyOf((Collection < String >) o));
                break;
            case CI:
                manifestCiWriter.writeCiReferenceProperty(cache.lookup((ConfigurationItem) o));
                break;
            case SET_OF_CI:
            case LIST_OF_CI:
                if (pd.isAsContainment()) {
                    writeNestedConfigurationItems((Collection<ConfigurationItem>) o);
                } else {
                    manifestCiWriter.writeCiReferenceCollectionProperty(ImmutableList.copyOf(Collections2.transform((Collection<ConfigurationItem>)o, new Function<ConfigurationItem, String>() {
                        @Override
                        public String apply(ConfigurationItem input) {
                            return cache.lookup(input);
                        }
                    })));
                }
                break;
            case MAP_STRING_STRING:
                manifestCiWriter.writeMapStringStringProperty((Map<String, String>) o);
                break;
        }
        manifestCiWriter.endProperty();
    }

    private void writeNestedConfigurationItems(Collection<ConfigurationItem> cis) {
        for (ConfigurationItem ci : cis) {
            ManifestWriter.ManifestCiWriter ciWriter;
            if (ci instanceof Artifact) {
                ciWriter = writer.writeCi(cache.lookup(ci), ci.getType(), id(cache.lookup(ci), ((Artifact) ci).getFile().getName()));
            } else {
                ciWriter = writer.writeCi(cache.lookup(ci), ci.getType());
            }
            writeProperties(ci, ciWriter);
            ciWriter.endCi();
        }
    }

    @Override
    public String toString() {
        return writer.toString();
    }
}
