package com.xebialabs.deployit.engine.tasker

import akka.actor._
import com.xebialabs.deployit.engine.api.execution.{BlockExecutionState, StepExecutionState}
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.BlockStateChanged
import com.xebialabs.deployit.engine.tasker.messages.Stop
import akka.actor.Status.Failure
import com.xebialabs.deployit.engine.tasker.StepExecutingActor.messages.ExecuteStep
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.BlockDone
import com.xebialabs.deployit.engine.tasker.messages.Start
import com.xebialabs.deployit.engine.tasker.messages.Abort

object StepBlockExecutingActor {
  def props(task: Task, block: StepBlock, ctx: TaskExecutionContext) = Props(classOf[StepBlockExecutingActor], task, block, ctx)
}

class StepBlockExecutingActor(task: Task, sb: StepBlock, ctx: TaskExecutionContext) extends BaseExecutionActor {

  import context._
  import BlockExecutionState._

  var block: StepBlock = sb
  val taskId = task.getId

//  val stepActor = createChild(StepExecutingActor.props(task, ctx), "step")

  def receive: Actor.Receive = ReceiveWithMdc(task) {
    case Start(`taskId`) if Set(PENDING, STOPPED, ABORTED, FAILED).contains(block.state) =>
      updateStateAndNotify(BlockExecutionState.EXECUTING)
      executeStep(0, createChild(StepExecutingActor.props(task, ctx)))
    case Start(`taskId`) if block.state == DONE =>
      debug(s"Not Executing block [${block.id}] again as it is already [EXECUTED]")
      notifyAndDie(block.state, None)
    case Start(`taskId`) => sender ! Failure(new IllegalStateException(s"Can not execute a step block in state [${block.state}]"))
    case m @ _ =>
      error(s"Don't know what to do with [$m]")
      sender ! Failure(new IllegalStateException(s"$m"))
  }

  def executeStep(stepNr: Int, stepActor: ActorRef) {
    if (block.steps.size <= stepNr) {
      notifyAndDie(BlockExecutionState.DONE, Option(stepActor))
    } else {
      val head = block.steps(stepNr).asInstanceOf[TaskStep]
      become(waitForStep(head, stepNr + 1, stepActor))
      stepActor ! ExecuteStep(head, ctx.stepContext(head, task))
    }
  }

  def waitForStep(step: TaskStep, nextStepNr: Int, stepActor: ActorRef): Actor.Receive = ReceiveWithMdc(task) {
    case StepExecutionState.DONE | StepExecutionState.SKIPPED => executeStep(nextStepNr, stepActor)
    case StepExecutionState.PAUSED => notifyAndDie(BlockExecutionState.STOPPED, Option(stepActor))
    case StepExecutionState.FAILED => notifyAndDie(BlockExecutionState.FAILED, Option(stepActor))
    case Stop(`taskId`) =>
      info(s"Received [Stop] message")
      updateStateAndNotify(BlockExecutionState.STOPPING)
      become(stopAfterCurrentStep(step, nextStepNr, stepActor))
    case Abort(`taskId`) =>
      doAbort(stepActor, step)
  }


  def doAbort(stepActor: ActorRef, step: TaskStep) {
    info(s"Received [Abort] message")
    updateStateAndNotify(BlockExecutionState.ABORTING)
    context.watch(stepActor)
    become(aborting(step))
    step.interruptRunner()
    stepActor ! Kill
  }

  def aborting(step: TaskStep): Actor.Receive = ReceiveWithMdc(task) {
    case m @ Terminated(stepActor) =>
      info(s"Received [$m], going to aborted state")
      step.setState(StepExecutionState.FAILED)
      notifyAndDie(BlockExecutionState.ABORTED, None)
    case m @ _ =>
      info(s"Received [$m], ignoring")
  }

  def notifyAndDie(state: BlockExecutionState, stepActorOption: Option[ActorRef]) {
    if (state == DONE) {
      self ! PoisonPill
    } else {
      become(receive)
      stepActorOption.foreach(context.stop)
    }

    updateStateAndNotify(state)
    parent ! BlockDone(task.getId, block)
  }

  def stopAfterCurrentStep(step: TaskStep, nextStepNr: Int, stepActor: ActorRef): Actor.Receive = ReceiveWithMdc(task) {
    case Abort(`taskId`) =>
      doAbort(stepActor, step)
    case StepExecutionState.DONE if block.steps.size <= nextStepNr => notifyAndDie(BlockExecutionState.DONE, Option(stepActor))
    case m @ _ => notifyAndDie(BlockExecutionState.STOPPED, Option(stepActor))
  }

  def updateStateAndNotify(newState: BlockExecutionState) {
    val oldState: BlockExecutionState = block.state
    if (oldState != newState) {
      block = block.newState(newState)
      val msg: BlockStateChanged = BlockStateChanged(task.getId, block, oldState, newState)
      debug(s"Sending BlockStateChange($oldState->$newState) message for [${block.id}]")
      parent ! msg
    }
  }

}
