package com.xebialabs.deployit.engine.tasker

import akka.actor._
import com.xebialabs.deployit.engine.tasker.BlockExecutingActor.{BlockDone, BlockStateChanged}
import com.xebialabs.deployit.engine.tasker.messages.{Abort, Start, Stop}

object BlockExecutingActor {
  def props(task: Task, block: CompositeBlock, ctx: NestedTaskExecutionContext) = {
    if (block.isInstanceOf[StepBlock]) throw new IllegalArgumentException("I cannot handle Step Blocks...")
    Props(classOf[BlockExecutingActor], task, block, ctx)
  }

  case class BlockDone(taskId: TaskId, block: Block)

  case class BlockStateChanged(taskId: TaskId, block: Block, oldState: BlockState, state: BlockState)

}

class BlockExecutingActor(task: Task, b: CompositeBlock, ctx: NestedTaskExecutionContext) extends BaseExecutionActor {

  import context._
  import BlockState._

  var block: CompositeBlock = b
  val taskId = task.getId

  def updateStateAndNotify(newState: BlockState) {
    val oldState: BlockState = block.state
    if (oldState != newState) {
      block = block.newState(newState)
      val msg: BlockStateChanged = BlockStateChanged(task.getId, block, oldState, newState)
      log.debug(s"Sending BlockStateChange($oldState->$newState) message for [${block.id}]")
      parent ! msg
    }
  }

  def receive: Actor.Receive = {
    case Start(`taskId`) if Set(PENDING, STOPPED, ABORTED, FAILED).contains(block.state) => block match {
      case parallelBlock @ ParallelBlock(id, state, blocks) =>
        log.info(s"Processing ParallelBlock[$id] of task [${task.getId}]")
        updateStateAndNotify(BlockState.EXECUTING)
        executeParallelBlock(blocks)
      case serialBlock @ SerialBlock(id, state, blocks) =>
        log.info(s"Processing SerialBlock[$id] of task [${task.getId}]")
        updateStateAndNotify(BlockState.EXECUTING)
        executeSerialBlock(blocks)
    }

  }

  def executeParallelBlock(blocks: Seq[ImmutableBlock]) {
    val blocksWithActors: Seq[(ImmutableBlock, ActorRef)] = blocks.filterNot(_.state == EXECUTED).map(b => (b, createChildForBlock(b, ctx.contextFor(b))))
    become(waitForParallelBlocksCompleted(blocksWithActors))
    blocksWithActors.foreach {
      case (block, actor) => actor ! Start(taskId)
    }
  }

  def executeSerialBlock(blocks: Seq[ImmutableBlock]) {
    val head: ImmutableBlock = blocks.head
    head.state match {
      case EXECUTED =>
        executeSerialBlock(blocks.tail)
      case _ =>
        val child: ActorRef = createChildForBlock(head, ctx)
        become(waitForSerialBlockCompleted(blocks, child))
        child ! Start(taskId)
    }
  }

  def createChildForBlock(b: ImmutableBlock, ctx: NestedTaskExecutionContext): ActorRef = child(b.id) match {
    case Some(ref) => ref
    case None => b match {
      case sb: StepBlock => createChild(StepBlockExecutingActor.props(task, sb, ctx), sb.id)
      case cb: CompositeBlock => createChild(BlockExecutingActor.props(task, cb, ctx), cb.id)
    }
  }

  def waitForParallelBlocksCompleted(blocks: Seq[(ImmutableBlock, ActorRef)]): Actor.Receive = {
    case BlockDone(`taskId`, doneBlock: ImmutableBlock) =>
      log.debug(s"Received BlockDone message for [${doneBlock.id}]")
      val withoutCompletedBlock: Seq[(ImmutableBlock, ActorRef)] = blocks.filterNot(_._1.id == doneBlock.id)
      block = block.replaceBlock(doneBlock)
      updateStateAndNotify(block.determineNewState(doneBlock, withoutCompletedBlock.map(_._1)))
      if (withoutCompletedBlock.isEmpty) {
        notifyAndDie()
      } else {
        become(waitForParallelBlocksCompleted(withoutCompletedBlock))
      }
    case BlockStateChanged(`taskId`, changedBlock: ImmutableBlock, oldState, state) =>
      log.debug(s"Received BlockStateChange($oldState->$state) message for [${block.id}]")
      block = block.replaceBlock(changedBlock)
      updateStateAndNotify(block.determineNewState(changedBlock, blocks.map(_._1)))
      become(waitForParallelBlocksCompleted(blocks))
    case s @ Stop(`taskId`) =>
      log.debug(s"Received Stop($taskId) message, stopping sub-block")
      blocks.foreach(t => t._2 ! s)
    case m @ Abort(`taskId`) =>
      log.debug(s"Received [$m] message, aborting sub-block")
      blocks.foreach(t => t._2 ! m)
  }

  def waitForSerialBlockCompleted(blocks: Seq[ImmutableBlock], currentBlockActor: ActorRef): Actor.Receive = {
    case BlockDone(`taskId`, doneBlock: ImmutableBlock) =>
      log.debug(s"Received BlockDone message for [${doneBlock.id}]")
      val withoutCompletedBlock: Seq[ImmutableBlock] = blocks.tail
      block = block.replaceBlock(doneBlock)
      updateStateAndNotify(block.determineNewState(doneBlock, withoutCompletedBlock))
      if (withoutCompletedBlock.isEmpty || doneBlock.state != EXECUTED) {
        notifyAndDie()
      } else {
        executeSerialBlock(withoutCompletedBlock)
      }
    case BlockStateChanged(`taskId`, changedBlock: ImmutableBlock, oldState, state) =>
      log.debug(s"Received BlockStateChange($oldState->$state) message for [${changedBlock.id}]")
      block = block.replaceBlock(changedBlock)
      updateStateAndNotify(block.determineNewState(changedBlock, blocks))
      become(waitForSerialBlockCompleted(blocks, currentBlockActor))
    case s @ Stop(`taskId`) =>
      log.debug(s"Received Stop($taskId) message, stopping sub-block")
      currentBlockActor ! s
    case m @ Abort(`taskId`) =>
      log.debug(s"Received [$m] message, aborting sub-block")
      currentBlockActor ! m

  }

  def notifyAndDie() {
    if (block.state == EXECUTED) {
      log.debug("All is [EXECUTED], seppuku it is...")
      self ! PoisonPill
    } else {
      become(receive)
    }
    log.info(s"All sub blocks of [${block.id}] completed, notifying parent")
    parent ! BlockDone(taskId, block)
  }

}
