package com.xebialabs.deployit.engine.tasker

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.api.services.Repository
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener
import scala.collection.mutable
import java.io.{PrintWriter, StringWriter, Serializable}
import grizzled.slf4j.Logger


class NestedTaskExecutionContext(parent: Option[NestedTaskExecutionContext], spec: TaskSpecification) extends Serializable {
  val attributes: Cache[String, AnyRef] = Cache()
  @transient var repository: Repository = null

  if (spec != null) {
    import collection.convert.wrapAll._
    spec.getListeners.foreach( l => attributes.put(l.getClass.getName, l))
  }

  val childContexts: Cache[BlockId, NestedTaskExecutionContext] = Cache()

  def contextFor(block: Block) = {
    childContexts.get(block.id).getOrElse({
      val c = new NestedTaskExecutionContext(Some(this), null)
      childContexts.put(block.id, c)
      c
    })
  }

  private[tasker] def listeners : Set[ExecutionStateListener] = {
    val collect: Set[ExecutionStateListener] = attributes.collect({case (n, l) if l.isInstanceOf[ExecutionStateListener] => l.asInstanceOf[ExecutionStateListener]}).toSet
    parent match {
      case None => collect
      case Some(p) => collect ++ p.listeners
    }
  }

  def clearChildContexts() = childContexts.clear()

  def getAttribute(name: String): AnyRef = attributes.get(name).getOrElse(parent match {
    case None => null
    case Some(p) => p.getAttribute(name)
  })

  def setAttribute(name: String, v: AnyRef) = attributes.put(name, v)

  def unsetAttribute(name: String) = attributes.remove(name)

  def stepContext(step: TaskStep) = new ExecutionContext {
    val ERROR_PREFIX = "[ERROR]: "

    val stepLogger = Logger.apply(step.getImplementation.getClass)
    def setAttribute(name: String, value: scala.AnyRef) {
      attributes.put(name, value)
    }

    def logOutput(output: String) {
      stepLogger.info(output)
      step.logBuilder.append(output).append("\n")
      step.touch()
    }

    def logError(error: String) {
      stepLogger.error(error)
      appendErrorToLog(error)
      step.touch()
    }

    def logError(error: String, t: Throwable) {
      stepLogger.error(error, t)
      appendErrorToLog(error)
      val stringWriter: StringWriter = new StringWriter
      t.printStackTrace(new PrintWriter(stringWriter))
      for (stackTraceLine <- stringWriter.toString.split("\n")) {
        appendErrorToLog(stackTraceLine)
      }
      step.touch()
    }

    def appendErrorToLog(error: String) {
      step.logBuilder.append(ERROR_PREFIX).append(error).append("\n")
    }

    def getAttribute(name: String): AnyRef = attributes.getOrElse(name, null)

    def getInspectionContext: InspectionContext = spec.getInspectionContext

    def getRepository: Repository = repository
  }
}
