package com.xebialabs.deployit.engine.tasker

import akka.actor._
import java.util
import java.util.UUID
import collection.convert.wrapAll._
import akka.util.Timeout
import grizzled.slf4j.Logging
import scala.concurrent.{Promise, Await}
import language.postfixOps
import scala.util.Try
import java.io._
import com.xebialabs.deployit.engine.tasker.messages._
import concurrent.duration._
import language.postfixOps
import akka.pattern._
import com.xebialabs.deployit.engine.tasker.messages.Stop
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.Register
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.Recovered
import scala.util.Failure
import scala.Some
import com.xebialabs.deployit.engine.tasker.messages.Archived
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.Cancel
import com.xebialabs.deployit.engine.tasker.messages.Cancelled
import scala.util.Success
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.Schedule
import com.xebialabs.deployit.engine.tasker.TaskManagingActor.messages.ArchiveTask
import com.xebialabs.deployit.engine.tasker.messages.Abort
import com.xebialabs.deployit.engine.spi.services.RepositoryFactory
import javax.annotation.{PostConstruct, PreDestroy}
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState2
import akka.actor.ActorDSL._
import java.util.concurrent.Semaphore
import com.xebialabs.deployit.engine.tasker
import com.xebialabs.deployit.engine.tasker.ArchivedListeningActor.Forward


class TaskExecutionEngine(archive: Archive, repositoryFactory: RepositoryFactory, implicit val system: ActorSystem) extends IEngine with Logging {
  def this(archive: Archive, repositoryFactory: RepositoryFactory) = this(archive, repositoryFactory: RepositoryFactory, ActorSystem("TaskExecutionEngine"))

  // Register the ArchiveActor
  val archiver = system.actorOf(ArchiveActor.props(archive), ArchiveActor.name)

  @PreDestroy
  def shutdownTasks() {
    import concurrent.duration._

    TaskRegistryExtension(system).getTasks.filter(t => Set(TaskExecutionState2.ABORTING, TaskExecutionState2.EXECUTING, TaskExecutionState2.FAILING, TaskExecutionState2.STOPPING).contains(t.state)).foreach {
      t =>
        info(s"Stopping task [${t.getId}] due to shutdown")
        lookupTaskActor(t.getId) ! Stop
        val promise = Promise[String]()
        actor(new Act {
          become {
            case TaskDone(doneTask) if doneTask.getId == t.getId => promise.success(doneTask.getId)
          }
        })
        Try(Await.result(promise.future, 10 seconds)) match {
          case Failure(exception) => warn(s"Failed to stop task [${t.getId}]", exception)
          case Success(value) => info(s"Successfully stopped task [$value]")
        }
    }

    system.shutdown()
  }

  @PostConstruct
  def recoverTasks() {
    val taskFiles: Array[File] = TaskerSettings(system).recoveryDir.listFiles(new FileFilter {
      def accept(p1: File): Boolean = p1.getName.endsWith(".task")
    })
    Option(taskFiles).getOrElse(Array[File]()).foreach { file =>
      Task(file) match {
        case Some(t) =>
          t.context.repository = repositoryFactory.create(t.getTempWorkDir)
          implicit val timeout = new Timeout(1 second)
          Await.ready(createTaskActor(t) ? Recovered(t), timeout.duration)
        case None =>
      }
    }
  }

  def archive(taskid: String) {
    val p = Promise[TaskId]()
    val listener: ActorRef = system.actorOf(ArchivedListeningActor.props(taskid, p))
    val message: AnyRef = ArchiveTask(taskid, archiver, listener)
    sendFinalMessageAndWaitForAnswer(p, taskid, message, listener)
  }


  def sendFinalMessageAndWaitForAnswer(p: Promise[TaskId], taskid: String, message: AnyRef, listener: ActorRef) {
    import scala.concurrent.ExecutionContext.Implicits.global
    val readySemaphore = new Semaphore(0)
    var caughtException: Option[TaskerException] = None
    p.future.onComplete {
      case Success(`taskid`) => readySemaphore.release()
      case Failure(ex) =>
        caughtException = Some(ex.asInstanceOf[TaskerException])
        readySemaphore.release()
    }
    val taskActor: ActorSelection = lookupTaskActor(taskid)
    listener ! Forward(taskActor, message)
    readySemaphore.acquire()
    system.stop(listener)
    if (caughtException.isDefined) {
      throw caughtException.get
    }
  }

  def addPauseStep(taskid: String, position: Int) {
    retrieve(taskid).addPause(position)
  }

  def moveStep(taskid: String, stepNr: Int, newPosition: Int) {
    retrieve(taskid).moveStep(stepNr, newPosition)
  }

  def unskipSteps(taskid: String, stepNrs: util.List[Integer]) {
    retrieve(taskid).unskip(stepNrs.map(_.intValue()).toList)
  }

  def skipSteps(taskid: String, stepNrs: util.List[Integer]) {
    retrieve(taskid).skip(stepNrs.map(_.intValue()).toList)
  }

  def cancel(taskid: String) {
    val p = Promise[TaskId]()
    val listener: ActorRef = system.actorOf(ArchivedListeningActor.props(taskid, p))
    val message: AnyRef = Cancel(taskid, archiver, listener)
    sendFinalMessageAndWaitForAnswer(p, taskid, message, listener)
  }

  def stop(taskid: String) {
    lookupTaskActor(taskid) ! Stop(taskid)
  }

  def abort(taskid: String) {
    lookupTaskActor(taskid) ! Abort(taskid)
  }

  def execute(taskid: String) {
    lookupTaskActor(taskid) ! Enqueue(taskid)
  }

  def schedule(taskid: String, scheduleAt: com.github.nscala_time.time.Imports.DateTime): Unit = {
    import com.github.nscala_time.time.Imports._
    def p(d: DateTime) = d.toString("yyyy-MM-dd HH:mm:ss Z")
    if (scheduleAt.isBeforeNow) {
      throw new TaskerException(s"Cannot schedule a task for the past, date entered was [${p(scheduleAt)}, now is [${p(DateTime.now)}]")
    }
    val delayMillis: Long = (DateTime.now to scheduleAt).millis
    val tickMillis: Long = TaskerSettings(system).tickDuration
    if (delayMillis > Int.MaxValue.toLong * tickMillis) {
      throw new TaskerException(s"Cannot schedule task [$taskid] at [${p(scheduleAt)}], because it is too far into the future. Can only schedule to [${p(new DateTime(DateTime.now.millis + (tickMillis * Int.MaxValue)))}]")
    }
    lookupTaskActor(taskid) ! Schedule(taskid, scheduleAt)
  }

  def register(spec: TaskSpecification): String = {
    def registerDefaultTriggers(spec: TaskSpecification) {
      spec.getListeners.add(new OldExecutionContextListenerCleanupTrigger)
    }

    registerDefaultTriggers(spec)
    val task: Task = new Task(UUID.randomUUID().toString, spec)
    task.context.repository = repositoryFactory.create(task.getTempWorkDir)
    implicit val timeout = new Timeout(1 second)
    Await.ready(createTaskActor(task) ? Register(task), timeout.duration)
    task.getId
  }

  def retrieve(taskid: String): Task = TaskRegistryExtension(system).getTask(taskid).getOrElse(throw new TaskNotFoundException("registry", taskid))

  def getAllIncompleteTasks: util.List[Task] = TaskRegistryExtension(system).getTasks

  protected[tasker] def lookupTaskActor(s: String): ActorSelection = system.actorSelection(system.child(s))

  protected[tasker] def createTaskActor(task: Task) = system.actorOf(TaskManagingActor.props, task.getId)

  def getSystem: ActorSystem = system
}

object ArchivedListeningActor {
  def props(taskId: TaskId, promise: Promise[TaskId]) = Props(classOf[ArchivedListeningActor], taskId, promise)
  case class Forward(actor: ActorSelection, message: AnyRef)
}

class ArchivedListeningActor(taskId: TaskId, promise: Promise[TaskId]) extends Actor {
  import context._

  def receive: Actor.Receive = {
    case Forward(actor, message) =>
      become(await)
      actor ! message
  }

  def await: Actor.Receive = {
    case FailedToArchive(`taskId`, exception) =>
      promise.failure(new TaskerException(exception, s"Task [$taskId] failed to archive"))
    case akka.actor.Status.Failure(exception) =>
      promise.failure(new TaskerException(exception, s"Task [$taskId] failed to archive"))
    case Archived(`taskId`) | Cancelled(`taskId`) =>
      promise.success(taskId)

  }
}
