package com.xebialabs.deployit.engine.tasker;

import java.io.Serializable;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.engine.api.execution.StepExecutionState;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.plugin.api.flow.Step;

import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.engine.api.execution.StepExecutionState.EXECUTING;
import static com.xebialabs.deployit.engine.api.execution.StepExecutionState.FAILED;
import static com.xebialabs.deployit.engine.api.execution.StepExecutionState.PENDING;
import static com.xebialabs.deployit.engine.api.execution.StepExecutionState.SKIP;

@SuppressWarnings("serial")
public class TaskStep implements StepState, Serializable {

    public static final String ERROR_PREFIX = "[ERROR]: ";

    private final Step implementation;
    private final Map<String,String> metadata = newHashMap();
    private volatile StepExecutionState state;
    private transient Thread runningThread;
    private AtomicInteger failureCount;
    StringBuilder logBuilder;
    private DateTime startDate;
    private DateTime completionDate;
    private DateTime lastModificationDate;

    public TaskStep(Step step) {
        this.implementation = step;
        this.state = StepExecutionState.PENDING;
        this.failureCount = new AtomicInteger();
        this.logBuilder = new StringBuilder();
    }

    @SuppressWarnings("rawtypes")
    public Step getImplementation() {
        return implementation;
    }

    void setState(StepExecutionState state) {
        this.state = state;
        if (state == FAILED) {
            failureCount.incrementAndGet();
        }
        touch();
    }

    void recordStart() {
        this.startDate = new DateTime();
        touch();
    }

    void recordCompletion() {
        this.completionDate = new DateTime();
        touch();
    }

    void clearLog() {
        logBuilder.setLength(0);
    }

    void touch() {
        this.lastModificationDate = new DateTime();
    }

    public boolean isMarkedForSkip() {
        return getState() == SKIP;
    }

    // FIXME Write test for Unskippable
    public boolean canSkip() {
        return (getState() == PENDING || getState() == FAILED);
    }

    public boolean hasExecuted() {
        return state == StepExecutionState.DONE || state == StepExecutionState.SKIPPED;
    }

    @Override
    public String toString() {
        return "[" + implementation.getDescription() + "]";
    }

    @Override
    public StepExecutionState getState() {
        return state;
    }

    @Override
    public String getLog() {
        return logBuilder.toString();
    }

    @Override
    public String getDescription() {
        return implementation.getDescription();
    }

    @Override
    public DateTime getStartDate() {
        return startDate;
    }

    @Override
    public DateTime getCompletionDate() {
        return completionDate;
    }

    public DateTime getLastModificationDate() {
        return lastModificationDate;
    }

    @Override
    public int getFailureCount() {
        return failureCount.intValue();
    }

    public void setRunner() {
        runningThread = Thread.currentThread();
    }

    public void interruptRunner() {
        if (runningThread != null) {
            runningThread.interrupt();
        }
    }

    @Override
    public Map<String, String> getMetadata() {
        return metadata;
    }

    public boolean isFailed() {
        return getState() == FAILED;
    }

    public TaskStep recovered() {
        if (state == EXECUTING) {
            setState(FAILED);
        }
        return this;
    }

    static final Logger logger = LoggerFactory.getLogger(TaskStep.class);
}
