package com.xebialabs.platform.watcher

import scalax.file.Path
import grizzled.slf4j.Logging
import com.xebialabs.platform.watcher.StateChange.StateChange
import com.xebialabs.platform.watcher.Existence.Existence

abstract class FileWatch(fileToWatch: Path) extends Logging {

  private var lastState: State = currentState

  type Notify = StateChange => Unit 
  
  private def currentState = State(currentFileExistence(), fileToWatch.lastModified)

  private def currentFileExistence() = fileToWatch.exists match {
    case true => Existence.Present
    case false => Existence.NotPresent
  }

  def doWhenModified(doNotify: Notify) {
    trace(s"Polling changes on file ${fileToWatch.path}")
    val newState = currentState
    if (isStateChanged(newState)) {
      val change = stateChange(newState)
      debug(s"State of the file ${fileToWatch.path} changed to $change")
      if(change == StateChange.NotChanged) {
        warn(s"State of the file ${fileToWatch.path} not expected to be $change")
      } else {
        doNotify(change)
      }
      lastState = newState
    }
  }

  private def isStateChanged(newState: State) = newState != lastState

  private def stateChange(newState: State) = lastState.existence match {
    case Existence.NotPresent =>
      newState.existence match {
        case Existence.Present => StateChange.Created
        case Existence.NotPresent => StateChange.NotChanged
      }
    case Existence.Present =>
      newState.existence match {
        case Existence.Present =>
          if (lastState.modified != newState.modified) StateChange.Modified else StateChange.NotChanged
        case Existence.NotPresent =>
          StateChange.Deleted
      }
  }

  private case class State(existence : Existence, modified : Long)
}

private[watcher] object Existence extends Enumeration {
  type Existence = Value
  val Present, NotPresent = Value
}

object StateChange extends Enumeration {
  type StateChange = Value
  val Created, Modified, Deleted, NotChanged = Value
}
