package com.xebialabs.deployit.booter.remote;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.concurrent.TimeUnit;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;

public class DeployitCommunicator {
    private final HttpClientHolder httpClientHolder;
    private final Proxies proxies;
    private BooterConfig config;

    public DeployitCommunicator(BooterConfig config) {
        this.config = config;
        this.httpClientHolder = new HttpClientHolder(config);
        tryAuthenticate(config);
        proxies = new Proxies(this);
    }

    public HttpClient httpClient() {
        return httpClientHolder.getHttpClient();
    }

    public HttpContext httpContext() {
        return httpClientHolder.getHttpContext();
    }

    public void shutdown() {
        logger.info("Shutting down {}", config);
        httpClientHolder.shutdown();
        // Callback to remove from the COMMUNICATORS map.
        RemoteBooter.remove(config);
        DescriptorRegistry.remove(config);
    }

    public Proxies proxies() {
        return proxies;
    }

    public Type getType(String typeName) {
        return DescriptorRegistry.getDescriptorRegistry(config).lookupType(typeName);
    }

    private void tryAuthenticate(BooterConfig config) {
        String urlToConnectTo = config.getUrl();
        logger.info("Connecting to the XL Deploy server at {}...", urlToConnectTo);
        try {
            HttpResponse execute = httpClient().execute(new HttpGet(urlToConnectTo + "/server/info"), httpContext());
            try {
                final int responseCode = execute.getStatusLine().getStatusCode();
                if (responseCode == 200) {
                    logger.info("Successfully connected.");
                } else if (responseCode == 401 || responseCode == 403) {
                    throw new IllegalStateException("You were not authenticated correctly, did you use the correct credentials?");
                } else if (responseCode == 402) {
                    throw new IllegalStateException("License not found, invalid, or expired; see the XL Deploy logs. Please contact your XebiaLabs sales representative for a valid license.");
                } else {
                    throw new IllegalStateException("Could contact the server at " + urlToConnectTo + " but received an HTTP error code, " + responseCode);
                }
            } finally {
                EntityUtils.consume(execute.getEntity());
                httpClient().getConnectionManager().closeIdleConnections(0, TimeUnit.MILLISECONDS);
            }
        } catch (MalformedURLException mue) {
            throw new IllegalStateException("Could not contact the server at " + urlToConnectTo, mue);
        } catch (IOException e) {
            throw new IllegalStateException("Could not contact the server at " + urlToConnectTo, e);
        }

    }

    public HttpClientHolder getHttpClientHolder() {
        return httpClientHolder;
    }

    public Proxies getProxies() {
        return proxies;
    }

    public BooterConfig getConfig() {
        return config;
    }

    private static final Logger logger = LoggerFactory.getLogger(DeployitCommunicator.class);
}
