package com.xebialabs.deployit.booter.remote;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistryId;
import com.xebialabs.deployit.plugin.api.reflect.Type;

import static java.lang.String.format;

public class RemoteDescriptorRegistry extends DescriptorRegistry {

    private final ReentrantReadWriteLock readWriteLock = new ReentrantReadWriteLock(true);
    private final Lock readLock = readWriteLock.readLock();
    private final Lock writeLock = readWriteLock.writeLock();

    private Map<Type, Descriptor> descriptors = new HashMap<Type, Descriptor>();
    private Map<Type, List<Type>> subtypes = newTypeMap();

    protected RemoteDescriptorRegistry(DescriptorRegistryId id) {
        super(id);
    }

    private Map<Type, List<Type>> newTypeMap() {
        Map<Type, List<Type>> map = new HashMap<Type, List<Type>>();
        return map;
    }

    private void addToTypeMap(Map<Type, List<Type>> newSubtypesMap, Type type, Type type2) {
        List<Type> collection = newSubtypesMap.get(type);
        if (null == collection) {
            collection = new ArrayList<Type>();
        }
        collection.add(type2);
        newSubtypesMap.put(type, collection);
    }

    private Collection<Type> getFromTypeMap(Map<Type, List<Type>> subtypesMap, Type supertype) {
        List<Type> list = subtypesMap.get(supertype);
        if (null == list) {
            list = new ArrayList<Type>();
        }
        return list;
    }


    @Override
    protected boolean isLocal() {
        return false;
    }

    @Override
    protected Collection<Descriptor> _getDescriptors() {
        readLock.lock();
        try {
            return descriptors.values();
        } finally {
            readLock.unlock();
        }
    }

    @Override
    protected Collection<Type> _getSubtypes(Type supertype) {
        readLock.lock();
        try {
            return getFromTypeMap(subtypes, supertype);
        } finally {
            readLock.unlock();
        }
    }


    @Override
    protected Descriptor _getDescriptor(Type type) {
        if (!descriptors.containsKey(type) ) {
            throw new IllegalArgumentException(format("DescriptorRegistry does not know about type [%s]", type));
        }
        readLock.lock();
        try {
            return descriptors.get(type);
        } finally {
            readLock.unlock();
        }
    }

    @Override
    protected boolean _exists(Type type) {
        readLock.lock();
        try {
            return descriptors.containsKey(type);
        } finally {
            readLock.unlock();
        }
    }

    public static void boot(DeployitCommunicator communicator) {
        RemoteDescriptorRegistry registry = new RemoteDescriptorRegistry(communicator.getConfig());
        DescriptorRegistry.add(registry);
        registry.reboot(communicator);
    }

    public void reboot(DeployitCommunicator communicator) {
        writeLock.lock();
        try {
            List<Descriptor> list = communicator.getProxies().getMetadataService().listDescriptors();
            reboot(list);
        } finally {
            writeLock.unlock();
        }
    }

    public void reboot(List<Descriptor> list) {
        writeLock.lock();
        try {
            Map<Type, Descriptor> newDescriptorsMap = new HashMap<Type, Descriptor>();
            Map<Type, List<Type>> newSubtypesMap = newTypeMap();

            for (Descriptor descriptor : list) {
                newDescriptorsMap.put(descriptor.getType(), descriptor);
                for (Type type : descriptor.getSuperClasses()) {
                    addToTypeMap(newSubtypesMap, type, descriptor.getType());
                }
                for (Type type : descriptor.getInterfaces()) {
                    addToTypeMap(newSubtypesMap, type, descriptor.getType());
                }
            }

            this.descriptors = newDescriptorsMap;
            this.subtypes = newSubtypesMap;
        } finally {
            writeLock.unlock();
        }
    }

    public Collection<Descriptor> getLoadedDescriptors() {
        return _getDescriptors();
    }

    public Descriptor getLoadedDescriptor(Type type) {
        return _getDescriptor(type);
    }

    public Descriptor getLoadedDescriptor(String prefixedName) {
        return getLoadedDescriptor(lookupType(prefixedName));
    }

    public Descriptor getLoadedDescriptor(String prefix, String name) {
        return getLoadedDescriptor(lookupType(prefix, name));
    }
}