package com.xebialabs.deployit.engine.tasker;

import java.io.PrintWriter;
import java.io.Serializable;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.engine.api.execution.StepExecutionState;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.services.Repository;

import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.engine.api.execution.StepExecutionState.*;

@SuppressWarnings("serial")
public class TaskStep implements StepState, Serializable {

    public static final String ERROR_PREFIX = "[ERROR]: ";

    private final Step implementation;
    private final Map<String,String> metadata = newHashMap();
    private volatile StepExecutionState state;
    private AtomicInteger failureCount;
    private StringBuilder logBuilder;
    private DateTime startDate;
    private DateTime completionDate;
    private DateTime lastModificationDate;

    public TaskStep(Step step) {
        this.implementation = step;
        this.state = StepExecutionState.PENDING;
        this.failureCount = new AtomicInteger();
        this.logBuilder = new StringBuilder();
    }

    @SuppressWarnings("rawtypes")
    public Step getImplementation() {
        return implementation;
    }

    void setState(StepExecutionState state) {
        this.state = state;
        if (state == FAILED) {
            failureCount.incrementAndGet();
        }
        touch();
    }

    void recordStart() {
        this.startDate = new DateTime();
        touch();
    }

    void recordCompletion() {
        this.completionDate = new DateTime();
        touch();
    }

    void clearLog() {
        logBuilder.setLength(0);
    }

    private void touch() {
        this.lastModificationDate = new DateTime();
    }

    public boolean isMarkedForSkip() {
        return getState() == SKIP;
    }

    // FIXME Write test for Unskippable
    public boolean canSkip() {
        return (getState() == PENDING || getState() == FAILED);
    }

    @Override
    public String toString() {
        return "[" + implementation.getDescription() + "]";
    }

    @Override
    public StepExecutionState getState() {
        return state;
    }

    @Override
    public String getLog() {
        return logBuilder.toString();
    }

    @Override
    public String getDescription() {
        return implementation.getDescription();
    }

    @Override
    public DateTime getStartDate() {
        return startDate;
    }

    @Override
    public DateTime getCompletionDate() {
        return completionDate;
    }

    public DateTime getLastModificationDate() {
        return lastModificationDate;
    }

    @Override
    public int getFailureCount() {
        return failureCount.intValue();
    }

    @Override
    public Map<String, String> getMetadata() {
        return metadata;
    }

    ExecutionContext getContext(TaskExecutionContext ctx) {
        return new TaskStepExecutionContext(ctx);
    }

    class TaskStepExecutionContext implements ExecutionContext {

        private TaskExecutionContext taskContext;

        private Logger stepLogger;

        TaskStepExecutionContext(TaskExecutionContext taskContext) {
            this.taskContext = taskContext;
            this.stepLogger = LoggerFactory.getLogger(implementation.getClass());
        }

        @Override
        public synchronized void logOutput(String output) {
            stepLogger.info(output);
            logBuilder.append(output).append("\n");
            touch();
        }

        @Override
        public synchronized void logError(String error) {
            stepLogger.error(error);
            appendErrorToLog(error);
            touch();
        }

        @Override
        public synchronized void logError(String error, Throwable t) {
            stepLogger.error(error, t);
            appendErrorToLog(error);
            StringWriter stringWriter = new StringWriter();
            t.printStackTrace(new PrintWriter(stringWriter));
            for (String stackTraceLine : stringWriter.toString().split("\n")) {
                appendErrorToLog(stackTraceLine);
            }
            touch();
        }

        private void appendErrorToLog(String error) {
            logBuilder.append(ERROR_PREFIX).append(error).append("\n");
        }

        @Override
        public Object getAttribute(String name) {
            Object object = taskContext.getAttribute(name);
            stepLogger.debug("Getting value of attribute {}: {}", name, object);
            return object;
        }

        @Override
        public void setAttribute(String name, Object object) {
            stepLogger.debug("Setting value of attribute {}: {}", name, object);
            taskContext.setAttribute(name, object);
        }

        @Override
        public Repository getRepository() {
            return taskContext.getRepository();
        }

        @Override
        public InspectionContext getInspectionContext() {
            return taskContext.getInspectionContext();
        }
    }

    public boolean isFailed() {
        return getState() == FAILED;
    }

    static final Logger logger = LoggerFactory.getLogger(TaskStep.class);
}
