package com.xebialabs.deployit.test.deployment;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.deployment.planner.DeploymentPlanner;
import com.xebialabs.deployit.deployment.planner.DeploymentPlannerFactory;
import com.xebialabs.deployit.deployment.planner.Plan;
import com.xebialabs.deployit.engine.replacer.Placeholders;
import com.xebialabs.deployit.inspection.Inspector;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.test.support.TestExecutionContext;
import com.xebialabs.deployit.test.support.TestInspectionContext;

import static com.google.common.collect.Maps.newHashMap;
import static com.xebialabs.deployit.plugin.api.flow.StepExitCode.FAIL;
import static java.lang.String.format;

public class DeployitTester {

    private final DeploymentPlanner planner;
    private final Inspector inspector;
    private final InMemoryRepository repository;

    public DeployitTester() {
        this.repository = new InMemoryRepository();
        this.planner = new DeploymentPlannerFactory().planner(repository);
        this.inspector = new Inspector();
    }

    public DeployitTester(DeploymentPlanner planner, Inspector inspector, InMemoryRepository repository) {
        this.planner = planner;
        this.inspector = inspector;
        this.repository = repository;
    }

    /**
     * @deprecated Just use the default constructor.
     */
    @Deprecated
    public static DeployitTester build() {
        return new DeployitTester();
    }

    public List<Step> resolvePlan(DeltaSpecification spec) {
        return resolve(spec).getSteps();
    }

    public Plan resolve(DeltaSpecification spec) {
        Plan plan = planner.plan(spec);

        planLogger.info("Generated plan for {}:\n{}\n", firstCallerOnStack("com.xebialabs"), plan.writePlan(new StringWriter()));

        return plan;
    }

    private static String firstCallerOnStack(String packagePrefix) {
        StackTraceElement[] stackTrace = new Exception().getStackTrace();
        for (int i = stackTrace.length - 1; i >= 0; i--) {
            StackTraceElement frame = stackTrace[i];
            if (frame.getClassName().startsWith(packagePrefix)) {
                return frame.getMethodName();
            }
        }
        return "<unknown>";
    }

    /**
     * @deprecated Use {@link #runInspectionTask(ConfigurationItem)}.
     */
    @Deprecated
    public List<ConfigurationItem> inspect(ConfigurationItem item) {
        com.xebialabs.deployit.test.support.LoggingExecutionContext ctx = new com.xebialabs.deployit.test.support.LoggingExecutionContext(DeployitTester.class);
        try {
            return inspect(item, ctx);
        } finally {
            ctx.destroy();
        }
    }

    /**
     * @deprecated Use {@link #runInspectionTask(ConfigurationItem, TestExecutionContext)}.
     */
    @Deprecated
    public List<ConfigurationItem> inspect(ConfigurationItem item, com.xebialabs.deployit.plugin.api.execution.ExecutionContext ctx) {
        return inspector.inspect(item, ctx);
    }

    public List<ConfigurationItem> runInspectionTask(ConfigurationItem item) {
        TestExecutionContext executionContext = new TestExecutionContext();
        try {
            return runInspectionTask(item, executionContext);
        } finally {
            executionContext.destroy();
        }
    }

    public List<ConfigurationItem> runInspectionTask(ConfigurationItem item, TestExecutionContext executionContext) {
        TestInspectionContext inspectionContext = executionContext.getInspectionContext();
        // Create plan
        Inspector.inspect(item, inspectionContext);

        // Run inspection
        StepExitCode result = executePlan(inspectionContext.getSteps(), executionContext);
        if (result == FAIL) {
            throw new DeployitTesterException("Inspection of " + item + " failed.");
        }

        return new ArrayList<ConfigurationItem>(inspectionContext.getInspected().values());
    }

    @SuppressWarnings("rawtypes")
    public Deployed generateDeployed(Deployable d, Container c, Type deployedType) {
        return generateDeployed(d, c, deployedType, null);
    }

    @SuppressWarnings("rawtypes")
    public Deployed generateDeployed(Deployable d, Container c, Type deployedType, Map<String, String> placeholders) {
        Descriptor deployed = DescriptorRegistry.getDescriptor(deployedType);
        Deployed<Deployable, Container> configurationItem = deployed.newInstance();
        configurationItem.setDeployable(d);
        configurationItem.setContainer(c);

        configurationItem.setId(c.getId() + "/" + substringAfterLastSlash(d.getId()));

        Descriptor deployableDescriptor = DescriptorRegistry.getDescriptor(d.getType());
        for (PropertyDescriptor propertyDescriptor : deployed.getPropertyDescriptors()) {
            String name = propertyDescriptor.getName();
            PropertyDescriptor deployablePropertyDescriptor = deployableDescriptor.getPropertyDescriptor(name);
            if (deployablePropertyDescriptor != null) {
                if (propertyDescriptor.getName().equals("placeholders")) {
                    propertyDescriptor.set(configurationItem, newHashMap());
                } else {
                    propertyDescriptor.set(configurationItem, deployablePropertyDescriptor.get(d));
                }
            }
        }

        if (configurationItem instanceof DerivedArtifact<?>) {
            DerivedArtifact<?> da = (DerivedArtifact<?>) configurationItem;
            if (placeholders != null) {
                da.setPlaceholders(placeholders);
            }
            Placeholders.replacePlaceholders(da, new SimpleReplacer());
        }

        return configurationItem;
    }

    private static String substringAfterLastSlash(String id) {
        int i = id.lastIndexOf('/');
        if (i > -1) {
            return id.substring(i + 1);
        }
        return id;
    }

    public StepExitCode executePlan(List<Step> plan) {
        TestExecutionContext ctx = new TestExecutionContext(getClass());
        try {
            return executePlan(plan, ctx);
        } finally {
            ctx.destroy();
        }
    }

    public StepExitCode executePlan(List<Step> steps, ExecutionContext context) {
        StepExitCode result = StepExitCode.PAUSE;
        try {
            // Steps may be added dynamically
            for (int i = 0; i < steps.size(); i++) {
                result = steps.get(i).execute(context);
                if (result == StepExitCode.FAIL) {
                    break;
                }
            }
            return result;
        } catch (Exception e) {
            throw new DeployitTesterException(e);
        }
    }

    public InMemoryRepository getRepository() {
        return repository;
    }

    @SuppressWarnings("serial")
    public static class DeployitTesterException extends RuntimeException {
        DeployitTesterException(String message, Object... params) {
            super(format(message, params));
        }

        DeployitTesterException(Exception e) {
            super(e);
        }
    }

    private final static Logger planLogger = LoggerFactory.getLogger(Plan.class);
}

