package com.xebialabs.deployit.test.support;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.*;

import com.xebialabs.deployit.booter.local.utils.Strings;
import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;
import org.jdom2.input.sax.XMLReaders;

import com.google.common.base.CharMatcher;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Container;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.LIST_OF_CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.SET_OF_CI;

public class ItestTopology {

    private String id;

    private Set<String> tags = newHashSet();

    private boolean enabledByDefault = false;

    private Map<String, ConfigurationItem> items = newHashMap();

    private List<Container> targets = newArrayList();

    private Map<String, Map<String, String>> targetProperties = newHashMap();

    private Map<String, String> placeholders = newHashMap();

    private ItestTopology(Element topologyElement) {
        init(topologyElement);

    }

    public boolean isEnabledByDefault() {
        return enabledByDefault;
    }

    public String getId() {
        return id;
    }

    public Set<String> getTags() {
        return tags;
    }

    public List<Container> getTargets() {
        return targets;
    }

    public Map<String, Map<String, String>> getTargetProperties() {
        return targetProperties;
    }

    public Map<String, String> getPlaceholders() {
        return placeholders;
    }

    public Map<String, ConfigurationItem> getItems() {
        return items;
    }

    public String replacePlaceholders(String template) {
        for (String placeholder : placeholders.keySet()) {
            template = template.replace("${" + placeholder + "}", placeholders.get(placeholder));
        }
        return template;
    }

    public Map<String, String> getTargetProperties(String targetId) {
        Map<String, String> props = targetProperties.get(targetId);
        if (props == null) {
            props = Collections.emptyMap();
        }
        return props;
    }

    public String getTargetProperty(String targetId, String propertyName) {
        Map<String, String> props = targetProperties.get(targetId);
        if (props != null) {
            return props.get(propertyName);
        }
        return null;
    }

    public boolean hasMatchingTarget(Type type) {

        for (Container target : targets) {
            if (target.getType().instanceOf(type)) {
                return true;
            }
        }
        return false;
    }

    public boolean hasMatchingCi(Type type) {

        for (ConfigurationItem item : items.values()) {
            if (item.getType().instanceOf(type)) {
                return true;
            }
        }
        return false;
    }


    public ConfigurationItem findFirstMatchingCi(Type type) {

        for (ConfigurationItem item : items.values()) {
            if (item.getType().instanceOf(type)) {
                return item;
            }
        }

        throw new RuntimeException("Cannot resolve type " + type + " to a configuration item in the topology");
    }

    public Container findFirstMatchingTarget(Type type) {

        for (Container target : targets) {
            if (target.getType().instanceOf(type)) {
                return target;
            }
        }

        throw new RuntimeException("Cannot resolve type " + type + " to a container in the topology");
    }

    public static boolean isItestEnabled(String itestName, boolean defaultValue) {
        String enabledItestsSpecification = System.getProperty("itests");
        if (Strings.isBlank(enabledItestsSpecification)) {
            return defaultValue;
        }

        Iterable<String> enabledItests = Splitter.on(CharMatcher.WHITESPACE).split(enabledItestsSpecification);
        return Iterables.contains(enabledItests, itestName);
    }

    public static Map<String, ItestTopology> load() {
        return load(getTopologyItestFile());
    }

    public static Map<String, ItestTopology> load(File file) {
        Map<String, ItestTopology> topologies = newHashMap();

        SAXBuilder sb = new SAXBuilder(XMLReaders.NONVALIDATING);
        Document topologyXml = null;
        try {
            topologyXml = sb.build(file);
        } catch (JDOMException e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        List<Element> topElements = topologyXml.getRootElement().getChildren("toplogy");
        if (topElements.isEmpty()) {
            topElements = topologyXml.getRootElement().getChildren("topology");
        }
        for (Element topElement : topElements) {
            ItestTopology t = new ItestTopology(topElement);
            topologies.put(t.id, t);
        }

        return topologies;

    }

    private static File getTopologyItestFile() {
        URL url = Thread.currentThread().getContextClassLoader().getResource("ItestTopology.xml");
        checkNotNull(url);
        try {
            return new File(url.toURI());
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private void init(Element topologyElement) {
        id = Preconditions.checkNotNull(topologyElement.getAttributeValue("id"));
        String enabledString = topologyElement.getAttributeValue("enableByDefault");
        enabledByDefault = enabledString != null && enabledString.equals("true");

        String tagsString = topologyElement.getAttributeValue("tags");
        if (Strings.isNotBlank(tagsString)) {
            tags = newHashSet(Splitter.on(',').trimResults().split(tagsString));
        }

        Element configurationItemsElement = topologyElement.getChild("configurationItems");
        checkNotNull(configurationItemsElement, "configurationItems element required.");
        List<Element> ciElements = configurationItemsElement.getChildren();
        for (Element ciElement : ciElements) {
            registerCi(ciElement);
        }
        for (Element ciElement : ciElements) {
            registerCiProperties(ciElement);
        }

        Element targetsElement = topologyElement.getChild("targets");
        if (targetsElement != null) {
            List<Element> targetElements = targetsElement.getChildren();
            for (Element targetElement : targetElements) {
                registerTarget(targetElement);
            }
        }

        Element placeholdersElement = topologyElement.getChild("placeholders");
        if (placeholdersElement != null) {
            List<Element> placeholderElements = placeholdersElement.getChildren();
            for (Element pElement : placeholderElements) {
                this.placeholders.put(pElement.getName(), Preconditions.checkNotNull(pElement.getValue()).trim());
            }
        }

        resolveParentChildRelationships();
    }

    private void resolveParentChildRelationships() {
        for (ConfigurationItem item : items.values()) {
            resolveParentChildRelationship(item);
        }

    }

    private void registerTarget(Element targetElement) {
        ConfigurationItem target = resolveCiReference(Preconditions.checkNotNull(targetElement.getAttributeValue("id"), "id is required for target element"), targetElement.getName());
        Descriptor d = DescriptorRegistry.getDescriptor(target.getType());
        checkArgument(d.isAssignableTo(Container.class), "Type [%s] is not assignable to udm.Container", d.getType());
        targets.add((Container) target);

        for (Element p : targetElement.getChildren()) {
            if (!targetProperties.containsKey(target.getId())) {
                targetProperties.put(target.getId(), new HashMap<String, String>());
            }

            Map<String, String> props = targetProperties.get(target.getId());
            props.put(p.getName(), Preconditions.checkNotNull(p.getValue()).trim());
        }
    }

    private void registerCi(Element ciElement) {
        Descriptor d = DescriptorRegistry.getDescriptor(ciElement.getName());
        String id = checkNotNull(ciElement.getAttributeValue("id"), "Element [%s] has no id attribute", ciElement.getName());
        ConfigurationItem item = d.newInstance(id);
        items.put(item.getId(), item);
    }

    private void registerCiProperties(Element ciElement) {
        Descriptor d = DescriptorRegistry.getDescriptor(ciElement.getName());

        String id = ciElement.getAttributeValue("id");
        ConfigurationItem item = items.get(id);

        for (Element p : ciElement.getChildren()) {
            PropertyDescriptor pd = Preconditions.checkNotNull(d.getPropertyDescriptor(p.getName()), "Property [%s.%s] does not exist", d.getType(), p.getName());
            switch (pd.getKind()) {
                case CI:
                    pd.set(item, resolveCiReference(p));
                    break;
                case MAP_STRING_STRING:
                    Map<String, String> map = newHashMap();
                    for (Element e : p.getChildren("entry")) {
                        map.put(Preconditions.checkNotNull(e.getAttributeValue("key")), Preconditions.checkNotNull(e.getValue()));
                    }
                    pd.set(item, map);
                    break;
                case SET_OF_STRING:
                    Set<String> setOfString = newHashSet();
                    pd.set(item, extractElementValuesIntoCollection(p,setOfString));
                    break;
                case LIST_OF_STRING:
                    List<String> listOfString = newArrayList();
                    pd.set(item, extractElementValuesIntoCollection(p,listOfString));
                    break;
                case SET_OF_CI:
                    Set<ConfigurationItem> setOfCi = newHashSet();
                    pd.set(item, extractElementRefValuesIntoCollection(p, setOfCi));
                    break;
                case LIST_OF_CI:
                    List<ConfigurationItem> listOfCi = newArrayList();
                    pd.set(item, extractElementRefValuesIntoCollection(p, listOfCi));
                    break;
                default:
                    pd.set(item, p.getValue());
            }
        }
    }

    private Collection<ConfigurationItem> extractElementRefValuesIntoCollection(Element p, Collection<ConfigurationItem> col) {
        for (Element v : (List<Element>) p.getChildren("value")) {
            col.add(resolveCiReference(v));
        }
        return col;
    }

    private Collection<String> extractElementValuesIntoCollection(Element p, Collection<String> col) {
        for (Element v : (List<Element>) p.getChildren("value")) {
            col.add(v.getValue());
        }
        return col;
    }


    private ConfigurationItem resolveCiReference(Element p) {
        return resolveCiReference(p.getValue(), p.getName());
    }

    private ConfigurationItem resolveCiReference(String id, String elementName) {
        String ciRefId = id.trim();
        ConfigurationItem ciRef = items.get(ciRefId);
        return checkNotNull(ciRef, "Cannot resolve ci ref [%s] for element [%s]", ciRefId, elementName);
    }

    @SuppressWarnings("unchecked")
    private void resolveParentChildRelationship(ConfigurationItem item) {
        Descriptor d = DescriptorRegistry.getDescriptor(item.getType());
        for (PropertyDescriptor pd : d.getPropertyDescriptors()) {
            boolean isReferenceToParent = pd.getKind() == CI && pd.isAsContainment();
            if (isReferenceToParent) {
                ConfigurationItem parent = getParent(item, pd);
                pd.set(item, parent);

                for (PropertyDescriptor ppd : parent.getType().getDescriptor().getPropertyDescriptors()) {
                    boolean isReferenceToChild = (ppd.getKind() == SET_OF_CI || ppd.getKind() == LIST_OF_CI) && ppd.isAsContainment() && d.isAssignableTo(ppd.getReferencedType());
                    if (isReferenceToChild) {
                        Collection<ConfigurationItem> refs = (Collection<ConfigurationItem>) ppd.get(parent);
                        refs.add(item);
                        ppd.set(parent, refs);
                    }
                }
            }
        }
    }

    private ConfigurationItem getParent(ConfigurationItem item, PropertyDescriptor pd) {
        String parentId = getParentId(item.getId());
        ConfigurationItem parent;
        if (parentId.equals(item.getId())) {
            parent = item;
        } else {
            parent = items.get(parentId);
            checkArgument(parent != null, "Cannot resolve parent reference from item [%s] for property [%s]", item.getId(), pd);
        }
        return parent;
    }

    private String getParentId(String id) {
        int indexOfLastSlash = id.lastIndexOf('/');
        checkArgument(indexOfLastSlash != -1, "[%s] has no parent", id);
        return id.substring(0, indexOfLastSlash);
    }

}
