package com.xebialabs.deployit.test.support;

import java.util.HashMap;
import java.util.Map;

import com.google.common.collect.Sets;

import com.xebialabs.overcast.host.CloudHost;

public class CloudHostRegistry {

    private static final CloudHostRegistry instance = new CloudHostRegistry();

    private  Map<String, StatefulHost> registry = new HashMap<String, StatefulHost>();

    private CloudHostRegistry() {}

    public static CloudHostRegistry getInstance() {
        return instance;
    }

    public void makeSureIsUp(CloudHost cloudHost) {
        if (isRegisteredAndRunning(cloudHost.getHostName())) {

            StatefulHost statefulHost = registry.get(cloudHost.getHostName());
            if (statefulHost.state == StatefulHost.State.BROKEN) {
                throw new HostIsBrokenException("The cloud host which you are trying to put up is broken.");
            }

            if (statefulHost.state == StatefulHost.State.STARTING) {
                throw new HostIsStillStartingException("The cloud host which you are trying to put up is still starting.");
            }

            return;
        }

        try {
            registry.put(cloudHost.getHostName(), StatefulHost.starting(cloudHost));
            cloudHost.setup();
            registry.put(cloudHost.getHostName(), StatefulHost.up(cloudHost));
        } catch (RuntimeException e) {
            registry.put(cloudHost.getHostName(), StatefulHost.broken(cloudHost));
            throw e;
        }

    }

    public void makeSureIsDown(CloudHost cloudHost) {
        if (!isRegisteredAndRunning(cloudHost.getHostName())) {
            return;
        }
        cloudHost.teardown();
        registry.put(cloudHost.getHostName(), StatefulHost.down(cloudHost));
    }

    public synchronized void makeSureAllAreDown() {
        for (StatefulHost entry : registry.values()) {
            makeSureIsDown(entry.host);
        }
    }

    public void makeSureAllDownInTheEnd() {
        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                makeSureAllAreDown();
            }
        });
    }

    public void reset() {
        registry = new HashMap<String, StatefulHost>();
    }

    private boolean isRegisteredAndRunning(String key) {
        StatefulHost entry = registry.get(key);
        return entry != null && Sets.immutableEnumSet(StatefulHost.State.UP, StatefulHost.State.BROKEN, StatefulHost.State.STARTING).contains(entry.state);
    }

    private static class StatefulHost {
        private CloudHost host;
        private enum State {STARTING, UP, BROKEN, DOWN}

        private State state;

        private StatefulHost(CloudHost host, State state) {
            this.host = host;
            this.state = state;
        }

        static StatefulHost starting(CloudHost host) {
            return new StatefulHost(host, State.STARTING);
        }

        static StatefulHost up(CloudHost host) {
            return new StatefulHost(host, State.UP);
        }

        static StatefulHost down(CloudHost host) {
            return new StatefulHost(host, State.DOWN);
        }

        static StatefulHost broken(CloudHost host) {
            return new StatefulHost(host, State.BROKEN);
        }
    }

    @SuppressWarnings("serial")
    public static class HostIsBrokenException extends RuntimeException {
        public HostIsBrokenException(final String message) {
            super(message);
        }
    }

    @SuppressWarnings("serial")
    public static class HostIsStillStartingException extends RuntimeException {
        public HostIsStillStartingException(final String message) {
            super(message);
        }
    }

}
