package com.xebialabs.deployit.inspection;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentExecutionContext;
import com.xebialabs.deployit.plugin.api.execution.ExecutionContext;
import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.api.inspection.InspectionExecutionContext;
import com.xebialabs.deployit.plugin.api.inspection.InspectionPlanningContext;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

class InspectionContext implements InspectionExecutionContext, InspectionPlanningContext, DeploymentExecutionContext {

	private ExecutionContext context;

	private Map<String, Object> attributes;
	private Logger stepLogger;

	@SuppressWarnings("rawtypes")
    private List<Step> steps = newArrayList();
	private AtomicInteger currentStepIndex = new AtomicInteger(0);
	private List<ConfigurationItem> discovered = newArrayList();
	private List<ConfigurationItem> inspected = newArrayList();
	
	private List<String> capturedError = newArrayList();


	public InspectionContext(final ExecutionContext context) {
		this.context = checkNotNull(context);
	}

	public InspectionContext(final Map<String, Object> attributes) {
		this.attributes = checkNotNull(attributes);
	}

	public void startStepExecution(Step<?> step) {
		if(attributes != null) {
			this.stepLogger = LoggerFactory.getLogger(step.getClass());
		}
    }

	@Override
	public void addStep(@SuppressWarnings("rawtypes") Step step) {
		this.steps.add(step);
	}

	@Override
	public void discovered(ConfigurationItem item) {
		this.discovered.add(item);
	}
	
	void inspected(ConfigurationItem item) {
		this.discovered.remove(item);
		this.inspected.add(item);
	}

	@Override
	public void logOutput(String output) {
		if(context != null) {
			context.logOutput(output);
		}
		if(stepLogger != null) {
			stepLogger.info(output);
		}
	}

	@Override
	public void logError(String error) {
		if(context != null) {
			context.logError(error);
		}
		if(stepLogger != null) {
			stepLogger.error(error);
		}
    	this.capturedError.add(error);
	}

	@Override
	public void logError(String error, Throwable t) {
		if(context != null) {
			context.logError(error, t);
		}
		if(stepLogger != null) {
			stepLogger.error(error, t);
		}
    	this.capturedError.add(error);
    	if (t != null) {
            StringWriter sw = new StringWriter();
            t.printStackTrace(new PrintWriter(sw));
            this.capturedError.add(sw.toString());
        }
	}

	@Override
	public Object getAttribute(String name) {
		if(context != null) {
			return context.getAttribute(name);
		} else if(attributes != null) {
			return attributes.get(name);
		} else {
			throw new IllegalStateException(this + " has no context and no attributes");
		}
	}

	@Override
	public void setAttribute(String name, Object object) {
		if(context != null) {
			context.setAttribute(name, object);
		}
		if(attributes != null) {
			attributes.put(name, object);
		}
	}

	List<ConfigurationItem> getDiscovered() {
		return discovered;
	}

	public List<ConfigurationItem> getInspected() {
		return inspected;
	}

	@SuppressWarnings("rawtypes")
    public List<Step> getSteps() {
		return steps;
	}

	@SuppressWarnings("rawtypes")
    public Step getNextStep() {
		if (currentStepIndex.get() < steps.size()) {
			return steps.get(currentStepIndex.getAndIncrement());
		}
		return null;
	}

    public List<String> getCapturedError() {
        return capturedError;
    }

}
