/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.ci;

import static com.google.common.base.Preconditions.checkArgument;
import static java.lang.String.format;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.xebialabs.deployit.BaseConfigurationItem;
import com.xebialabs.deployit.ConfigurationItem;
import com.xebialabs.deployit.ConfigurationItemDescription;
import com.xebialabs.deployit.ConfigurationItemProperty;

@SuppressWarnings("serial")
@ConfigurationItem(description = "A generic web server.", category = "middleware")
public abstract class HttpdServer extends BaseConfigurationItem {

	@ConfigurationItemDescription
	protected String description;

	@ConfigurationItemProperty(required = true, description = "The host the web server is installed on")
	protected Host host;

	public static class HostAndPort {
		// "<host>:<port>" or "*:<port>" or "<host>:*"
		private static final Pattern HOST_PORT_PATTERN = Pattern.compile("(?:(\\S+|\\*)):(?:(\\d{1,5}+|\\*))");
		private final String host;
		private final String port;
		private final String filenameFriendlyHost;
		private final String filenameFriendlyPort;
		
		public HostAndPort(String hostAndPort) {
			checkArgument(hostAndPort != null, "Argument must be non-null");
			Matcher hostAndPortMatcher = HOST_PORT_PATTERN.matcher(hostAndPort);
			
			if (!isValidHostAndPort(hostAndPortMatcher)) {
				throw new IllegalArgumentException(format("'%s' does not match the pattern <hostname>:<port>", hostAndPort));
			}

			host = hostAndPortMatcher.group(1);
			port = hostAndPortMatcher.group(2);
			filenameFriendlyHost = toFilenameFriendlyName(host, "allhosts");
			filenameFriendlyPort = toFilenameFriendlyName(port, "allports");
		}
		
		private static String toFilenameFriendlyName(String value, String replacement) {
			return (value.equals("*") ? replacement : value);
		}
		
		public static boolean isValidHostAndPort(String hostAndPort) {
			if (hostAndPort == null) {
				return false;
			}
			return isValidHostAndPort(HOST_PORT_PATTERN.matcher(hostAndPort));
		}
		
		private static boolean isValidHostAndPort(Matcher hostAndPortMatcher) {
			return hostAndPortMatcher.matches();
		}
		
		public String getHost() {
			return host;
		}

		public String getPort() {
			return port;
		}
		
		public String getFilenameFriendlyHost() {
			return filenameFriendlyHost;
		}

		public String getFilenameFriendlyPort() {
			return filenameFriendlyPort;
		}		
		
	}
	
	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public Host getHost() {
		return host;
	}

	public void setHost(Host host) {
		this.host = host;
	}

}
