/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.hostsession.ssh;

import com.xebialabs.deployit.exception.RuntimeIOException;
import com.xebialabs.deployit.hostsession.CapturingCommandExecutionCallbackHandler;
import com.xebialabs.deployit.hostsession.CmdLine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.OutputStream;

/**
 * An output stream to a file on a host connected through SSH w/ SCP.
 */
class SshScpOutputStream extends OutputStream {

	protected SshScpHostFile file;

	protected long length;

    private SshScpProtocolHelper protocolHelper;


	SshScpOutputStream(SshScpHostFile file, long length) {
		this.file = file;
		this.length = length;
        this.protocolHelper = new SshScpProtocolHelper(file.sshHostSession, file.getPath());
	}

	void open() {
		try {
            protocolHelper.open("t");
            protocolHelper.readAck();
            protocolHelper.sendFilePreamble(file,length);
			logger.info("Opened SCP stream to write to remote file " + file);
		} catch (IOException exc) {
			throw new RuntimeIOException("Cannot open SCP stream to write remote file " + file, exc);
		}
	}

	@Override
	public void write(byte[] b) throws IOException {
		protocolHelper.write(b);
	}

	@Override
	public void write(byte[] b, int off, int len) throws IOException {
		protocolHelper.write(b, off, len);
	}

	@Override
	public void write(int b) throws IOException {
		protocolHelper.write(b);
	}

	@Override
	public void close() {
		try {
			protocolHelper.sendAck();
			protocolHelper.readAck();
		} catch (IOException ignore) {
		}

        int res = protocolHelper.close();
        if (res != 0) {
            throw new RuntimeIOException("Error closing SCP stream to write remote file " + file.remotePath + " (remote scp command returned error code " + res + ")");
        }
		chmodWrittenFile();
	}

	private void chmodWrittenFile() {
		if (file.sshHostSession instanceof SshSudoHostSession) {
			SshSudoHostSession sshSudoHostSession = (SshSudoHostSession) file.sshHostSession;
			CapturingCommandExecutionCallbackHandler capturedOutput = new CapturingCommandExecutionCallbackHandler();
            CmdLine chmod = new CmdLine().addArgument("chmod").addArgument("0666").addArgument(file.remotePath);
            int errno = sshSudoHostSession.noSudoExecute(capturedOutput, chmod);
			if (errno != 0) {
				throw new RuntimeIOException("Cannot chmod file " + file + ": " + capturedOutput.getError() + " (errno=" + errno + ")");
			}
			if (logger.isDebugEnabled())
				logger.debug("Chmodded file " + file);
		}
	}

	private Logger logger = LoggerFactory.getLogger(SshScpOutputStream.class);

}
