package com.xebialabs.deployit.plugin.freemarker

import java.io.{IOException, StringReader, StringWriter}

import com.xebialabs.deployit.plugin.freemarker.ConfigurationHolder._
import com.xebialabs.deployit.plugin.freemarker.sanitizer.OSScriptSanitizer
import com.xebialabs.overthere.RuntimeIOException
import freemarker.template.{Configuration, Template, TemplateException}
import grizzled.slf4j.Logging

private[freemarker] object Resolver extends Logging {

  val EXPRESSION_TOKEN = "${"
  val EXPRESSION_IF_TOKEN = "<#if"
  val NON_ESCAPED_COMMA = "(?<!\\\\),"


  def resolveExpression(expression: String, context: AnyRef, maskPassword: Boolean = false): String = {
    if (!isResolvable(expression)) {
      return expression
    }
    val cfg: Configuration = configuration(maskPassword)
    try {
      trace(s"Resolving expression $expression.")
      val template: Template = new Template("expression", new StringReader(expression), cfg)
      val sw: StringWriter = new StringWriter
      val mapWithSanitizers: AnyRef = addScriptSanitizers(context)
      template.process(convertToJavaMap(mapWithSanitizers), sw)
      sw.toString
    }
    catch {
      case e: IOException => throw new RuntimeIOException(e)
      case e: TemplateException => throw new RuntimeIOException(e)
    }
  }

  def resolveExpressions(expressions: Iterable[String], context: Map[String, AnyRef], maskPassword: Boolean = false): Iterable[String] = {
    transformCollectionByResolvingExpression(expressions, context, maskPassword)
  }

  def resolveMapExpressions(expressions: Map[String, String], context: Map[String, AnyRef], maskPassword: Boolean = false): Map[String, String] = {
    val contextWithExpressions: Map[String, AnyRef] = context ++ expressions
    expressions.mapValues(expr => resolveWhilePossible(expr, contextWithExpressions, maskPassword))
  }

  private def resolveWhilePossible(expression: String, context: AnyRef, maskPassword: Boolean): String = {
    if (isResolvable(expression))
      resolveWhilePossible(resolveExpression(expression, context, maskPassword), context, maskPassword)
    else
      expression
  }

  private def isResolvable(expression: String): Boolean = {
    expression != null && (expression.contains(EXPRESSION_TOKEN) || expression.contains(EXPRESSION_IF_TOKEN))
  }

  private[freemarker] def transformCollectionByResolvingExpression(expressions: Iterable[String], context: AnyRef, maskPassword: Boolean) = {
    expressions.map(exp => resolveExpression(exp, context, maskPassword))
      .flatMap(resolvedExpr => resolvedExpr.split(NON_ESCAPED_COMMA))
      .map(_.trim).filterNot(_.isEmpty)
      .map(expr => expr.replaceAll("\\\\,", ",")).toIterable
  }

  private[freemarker] def addScriptSanitizers(context: AnyRef) = context match {
    case map: Map[String, _] => {
      map + (OSScriptSanitizer.SANITIZE_EXPRESSION_UNIX -> OSScriptSanitizer.createUnixSpecificSanitizer(),
        OSScriptSanitizer.SANITIZE_EXPRESSION_WINDOWS -> OSScriptSanitizer.createWindowsSpecificSanitizer())
    }
    case _ => context
  }
}
