package com.xebialabs.deployit.plugin.bigip;

import java.util.Map;
import java.util.Set;
import com.google.common.collect.ImmutableMap;

import com.xebialabs.deployit.plugin.api.deployment.planning.DefaultOrders;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.generic.step.ScriptExecutionStep;
import com.xebialabs.deployit.plugin.lb.ci.LoadBalancer;
import com.xebialabs.deployit.plugin.overthere.Host;

import static java.lang.String.format;

@Metadata(root = Metadata.ConfigurationItemRoot.NESTED, virtual = false, description = "An F5 BigIp LoadBalancer")
public class LocalTrafficManager extends LoadBalancer {

    @Property(description = "The address of the BigIP load balancer")
    private String address;

    @Property(description = "The username used to connect to the BigIP load balancer")
    private String username;

    @Property(password = true, description = "The password used to connect to the BigIP load balancer")
    private String password;

    @Property(description = "The active partition on the BigIP load balancer")
    private String partition;

    @Property(defaultValue = "" + (DefaultOrders.STOP_ARTIFACTS - 2), hidden = true)
    private int removeFromPoolOrder = DefaultOrders.STOP_ARTIFACTS - 2;

    @Property(defaultValue = "big-ip/disable-pool-member.py.ftl", hidden = true)
    private String removeFromPoolScript = "big-ip/disable-pool-member.py.ftl";

    @Property(defaultValue = "" + (DefaultOrders.START_ARTIFACTS + 4), hidden = true)
    private int addToPoolOrder = DefaultOrders.START_ARTIFACTS + 4;

    @Property(defaultValue = "big-ip/enable-pool-member.py.ftl", hidden = true)
    private String addToPoolScript = "big-ip/enable-pool-member.py.ftl";

    @Property(description = "The host that runs the PyControl 2 library that controls the BigIP load balancer")
    private Host host;

    @Override
    public void stopTraffic(DeploymentPlanningContext ctx, Set<Container> serversToDisable) {
        for (Container container : serversToDisable) {
            ctx.addStep(new ScriptExecutionStep(removeFromPoolOrder, removeFromPoolScript, this, getFreeMarkerContext(container, this),
                    format("Remove %s from load balancer pool of %s", container, this)));
        }
    }

    @Override
    public void startTraffic(DeploymentPlanningContext ctx, Set<Container> serversToEnable) {
        for (Container container : serversToEnable) {
            ctx.addStep(new ScriptExecutionStep(addToPoolOrder, addToPoolScript, this, getFreeMarkerContext(container, this),
                    format("Return %s to load balancer pool of %s", container, this)));
        }
    }

    public Host getHost() {
        return host;
    }

    private static Map<String, Object> getFreeMarkerContext(Container container, Container loadBalancer) {
        return ImmutableMap.<String, Object>of("container", loadBalancer, "poolmember", container);
    }
}
