package com.xebialabs.deployit.plugin.cloud.step;

import java.util.List;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.cloud.ci.BaseHostTemplate;
import com.xebialabs.deployit.plugin.cloud.util.HostFactory;
import com.xebialabs.deployit.plugin.cloud.util.MarkerChecker;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.overthere.ConnectionOptions;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Step which waits for instances to become available based on SSH or marker file.
 * Depends on {@link ContextAttribute#CREATED_INSTANCES} and {@link ContextAttribute#USED_TEMPLATES} context attributes.
 */
@SuppressWarnings("serial")
public class WaitForInstancesStep implements Step {

    private MarkerChecker markerChecker;

    public WaitForInstancesStep(final MarkerChecker markerChecker) {
        this.markerChecker = markerChecker;
    }

    @Override
    public int getOrder() {
        return DEFAULT_ORDER + 10;
    }

    @Override
    public String getDescription() {
        return "Wait until created instances become available";
    }

    @Override
    @SuppressWarnings("unchecked")
    public StepExitCode execute(final ExecutionContext ctx) throws Exception {

        List<String> instances;
        List<BaseHostTemplate> templates;

        try {
            instances = checkNotNull((List<String>)ctx.getAttribute(ContextAttribute.CREATED_INSTANCES.toString()));
            templates = checkNotNull((List<BaseHostTemplate>)ctx.getAttribute(ContextAttribute.USED_TEMPLATES.toString()));
        } catch (NullPointerException e) {
            ctx.logError("Can not find required context attribute", e);
            return StepExitCode.FAIL;
        }

        for(int i=0; i < instances.size(); i++) {
            ctx.logOutput("Waiting until instance " + (i + 1) + " is booting...");

            BaseHostTemplate instanceTpl = templates.get(i);

            if (instanceTpl.getMarkerPath() != null) {
                Host host = HostFactory.fromTemplate(instanceTpl);
                String instancePublicAddress = instanceTpl.getInstanceIpAddress(instances.get(i));
                host.setProperty("address", instancePublicAddress);

                ConnectionOptions connectionOptions = new Host.ConnectionOptionsBuilder().getConnectionOptions(host);

                ctx.logOutput("Waiting until marker file " + instanceTpl.getMarkerPath() + " becomes available on " + instancePublicAddress + "...");
                markerChecker.waitForMarkerFileAppears(
                        (String) host.getProperty("protocol"),
                        connectionOptions,
                        instanceTpl.getMarkerPath(),
                        instanceTpl.getBootTimeout(),
                        instanceTpl.getRetryDelay()
                );
            }

        }

        return StepExitCode.SUCCESS;
    }
}

