package com.xebialabs.deployit.plugin.cloud.step;

import java.util.List;
import java.util.Map;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.cloud.step.rule.ResultCheck;
import com.xebialabs.deployit.plugin.cloud.util.CiParser;
import com.xebialabs.deployit.plugin.cloud.util.InstanceDescriptorResolver;

import freemarker.template.TemplateException;

import static com.google.common.collect.Maps.newHashMap;

@SuppressWarnings("serial")
public abstract class ValidateDescriptorStep implements Step {

    protected InstanceDescriptorResolver descriptorResolver;
    protected CiParser ciParser;
    protected Map<Object,Object> scope = newHashMap();
    private String descriptorTemplate;
    private String templateName;

    private ResultCheck resultCheck;

    public ValidateDescriptorStep(String descriptorTemplate, String templateName, final ResultCheck resultCheck) {
        this.descriptorResolver = new InstanceDescriptorResolver();
        this.ciParser = new CiParser();
        this.descriptorTemplate = descriptorTemplate;
        this.templateName = templateName;
        this.resultCheck = resultCheck;
    }

    @Override
    public String getDescription() {
        return "Validate XML descriptor for " + templateName;
    }


    @Override
    public int getOrder() {
        return DEFAULT_ORDER;
    }

    @Override
    public StepExitCode execute(final ExecutionContext ctx) throws Exception {

        String descriptor;

        try {
            descriptor = resolveTemplate(descriptorTemplate, scope, ctx);
            ctx.logOutput("Successfully processed descriptor template...");
            ctx.logOutput("-------------------------------------------");
            outputDescriptor(descriptor, ctx);
            ctx.logOutput("-------------------------------------------");
        } catch (Exception e) {
            ctx.logError("Can not process descriptor template.");
            ctx.logError(e.getMessage());
            return StepExitCode.FAIL;
        }

        try {
            List<ConfigurationItem> cis = parseCis(descriptor, ctx);
            ctx.logOutput("Successfully parsed " + cis.size() + " CIs");
            resultCheck.check(cis);
            return StepExitCode.SUCCESS;
        } catch (ResultCheck.Failed e) {
            ctx.logError("Resulted CIs don't meet expectation: " + e.getMessage());
            return StepExitCode.FAIL;
        } catch (Exception e) {
            ctx.logError("Can not parse generated XML. " + e.getMessage());
            return StepExitCode.FAIL;
        }

    }

    protected abstract String resolveTemplate(String tpl, Map<Object,Object> initialScope, ExecutionContext ctx) throws TemplateException;

    protected abstract List<ConfigurationItem> parseCis(String tpl, ExecutionContext ctx);

    /**
     * Outputs parsed descriptor to the context
     */
    protected void outputDescriptor(String descriptor, ExecutionContext ctx) {
        ctx.logOutput(descriptor);
    }

}
