package com.xebialabs.deployit.plugin.cmd.step;

import static com.google.common.base.Splitter.on;
import static com.google.common.base.Strings.nullToEmpty;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentExecutionContext;
import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep;
import com.xebialabs.deployit.plugin.api.execution.ExecutionContext;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.cmd.deployed.DeployedCommand;
import com.xebialabs.deployit.plugin.overthere.AutoFlushingExecutionContextOverthereProcessOutputHandler;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.ConnectionOptions;
import com.xebialabs.overthere.Overthere;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;

@SuppressWarnings("serial")
public class ExecuteCommandStep implements DeploymentStep {

    private DeployedCommand command;

    private transient OverthereConnection localConn;
    private transient OverthereConnection remoteConn;
    private transient ExecutionContext ctx;

    private Pattern placeHolderPattern = Pattern.compile("\\$\\{(.*)\\}");
	private int order;

	public ExecuteCommandStep(int order, DeployedCommand deployedCommand) {
		this.order = order;
		this.command = deployedCommand;
        validateCommandLineDependencyPlaceholders();
    }

	/*
	 * Cannot store the commandLineArgs Iterable as a field, as it is not serializable..
	 * (See: DEPLOYITPB-2234)
	 */
    private Iterable<String> getCommandLineArgs() {
        String params = command.getCommandLine();
        if (!nullToEmpty(params).trim().isEmpty()) {
            params = params.replace('\n', ' ');
            return on(' ').split(params);
        }
	    return newArrayList();
    }

    private void validateCommandLineDependencyPlaceholders() {
        for (String arg : getCommandLineArgs()) {
            String dependencyRef = extractDependencyRefFromPlaceholderArg(arg);
            if (dependencyRef != null) {
                boolean found = false;
                for (DeployableArtifact dependency : command.getDependencies()) {
                    if (dependency.getName().equals(dependencyRef)) {
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    throw new IllegalArgumentException("Placeholder arg " + arg + " does not refer to a valid dependency");
                }
            }
        }
    }

    private String extractDependencyRefFromPlaceholderArg(String arg) {
        Matcher matcher = placeHolderPattern.matcher(arg);
        if (matcher.matches()) {
            return matcher.group(1);
        }
        return null;
    }

    @Override
    public String getDescription() {
        return "Execute " + command.getName();
    }

    @Override
    public Result execute(DeploymentExecutionContext ctx) throws Exception {
        try {
            this.ctx = ctx;
            Map<String, OverthereFile> uploadedDependencies = uploadDependencies();
            CmdLine cmdLine = resolveCommandLine(uploadedDependencies);
            AutoFlushingExecutionContextOverthereProcessOutputHandler handler = new AutoFlushingExecutionContextOverthereProcessOutputHandler(ctx);
	        try {
	            ctx.logOutput("Executing command line : " + cmdLine.toCommandLine(command.getContainer().getOs(),true));
	            int rc = getRemoteConnection().execute(handler, cmdLine);
	            if (rc != 0) {
	                ctx.logError("Command failed with return code " + rc);
	                return Result.Fail;
	            }
            } finally {
            	handler.close();
            }
        } finally {
            disconnect();
        }

        return Result.Success;
    }

    private Map<String, OverthereFile> uploadDependencies() {
        Map<String, OverthereFile> uploadedFiles = newHashMap();
        if (command.getDependencies() == null || command.getDependencies().isEmpty()) {
            return uploadedFiles;
        }

        OverthereFile tempDir = getRemoteConnection().getTempFile("exec_cmd", ".tmp");
        tempDir.mkdir();
        ctx.logOutput("Uploading " + command.getDependencies().size() + " dependent files to " + tempDir.getPath());
        for (DeployableArtifact artifact : command.getDependencies()) {
            String name = artifact.getName();
            OverthereFile uploadedFile = tempDir.getFile(name);
            OverthereFile localFile = getLocalConnection().getFile(artifact.getFile().getPath());
            localFile.copyTo(uploadedFile);
            ctx.logOutput("Uploaded " + name);
            uploadedFiles.put(name, uploadedFile);
        }
        ctx.logOutput("Uploading done.");
        return uploadedFiles;
    }

    private CmdLine resolveCommandLine(Map<String, OverthereFile> dependencies) {
        CmdLine cmdLine = new CmdLine();
        for (String arg : getCommandLineArgs()) {
            String dependencyRef = extractDependencyRefFromPlaceholderArg(arg);
            if (dependencyRef != null) {
                cmdLine.addArgument(dependencies.get(dependencyRef).getPath());
            } else {
                cmdLine.addArgument(arg);
            }
        }
        return cmdLine;
    }

    private OverthereConnection getLocalConnection() {
        if (localConn == null) {
            localConn = Overthere.getConnection("local", new ConnectionOptions());
        }
        return localConn;
    }

    private OverthereConnection getRemoteConnection() {
        if (remoteConn == null) {
            remoteConn = command.getContainer().getConnection();
        }
        return remoteConn;
    }

    private void disconnect() {
        if (localConn != null)
            localConn.close();

        if (remoteConn != null)
            remoteConn.close();

        localConn = null;
        remoteConn = null;
    }

	@Override
	public int getOrder() {
		return order;
	}
}
