package com.xebialabs.deployit.plugins.releaseauth.command;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import nl.javadude.t2bus.Subscribe;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Sets;

import com.xebialabs.deployit.engine.spi.command.UpdateCiCommand;
import com.xebialabs.deployit.engine.spi.command.UpdateCisCommand;
import com.xebialabs.deployit.engine.spi.command.util.Update;
import com.xebialabs.deployit.engine.spi.event.DeployitEventListener;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Version;

import static com.google.common.collect.Sets.newHashSet;

@DeployitEventListener
public class RepositoryCommandListener {

    public static final String ADMIN = "admin";

    @Subscribe(canVeto = true)
    public void checkWhetherUpdateIsAllowed(UpdateCiCommand command) {
        checkUpdate(command.getUpdate(), newHashSet(command.getRoles()), command.getUsername());
    }

    private static void checkUpdate(final Update update, final Set<String> roles, final String username) {
        ConfigurationItem newCi = update.getNewCi();
        if (!newCi.getType().isSubTypeOf(Type.valueOf(Version.class))) {
            return;
        }

        final Descriptor descriptor = newCi.getType().getDescriptor();
        Collection<PropertyDescriptor> properties = descriptor.getPropertyDescriptors();
        FluentIterable<PropertyDescriptor> changed = getChangedProperties(update, getSatisfactionProperties(properties));
        FluentIterable<PropertyDescriptor> propertiesWithRolesLimitations = getPropertiesWithRolesLimitations(newCi, changed);
        FluentIterable<PropertyDescriptor> noPermission = getNoPermission(newCi, roles, username, descriptor, propertiesWithRolesLimitations);

        if (!noPermission.isEmpty()) {
            logger.error("User [{}] tried to update release conditions {} but didn't have permission for {} on ci {}", new Object[] { username, changed,
                noPermission, newCi.getId() });
            throw new CannotSetReleaseConditionsException(getPropertyNames(noPermission), newCi);
        }

        if (!changed.isEmpty()) {
            logger.info("User [{}] has updated release conditions {} on ci {}", new Object[] { username, changed, newCi.getId() });
        }
    }

    private static ImmutableList<String> getPropertyNames(FluentIterable<PropertyDescriptor> satisfiesForWhichUserHasNoPermission) {
        return satisfiesForWhichUserHasNoPermission.transform(new Function<PropertyDescriptor, String>() {
            public String apply(final PropertyDescriptor propertyDescriptor) {
                return propertyDescriptor.getName();
            }
        }).toImmutableList();
    }

    private static FluentIterable<PropertyDescriptor> getNoPermission(final ConfigurationItem ci, final Set<String> roles, final String username,
        final Descriptor descriptor, FluentIterable<PropertyDescriptor> properties) {
        return properties.filter(new Predicate<PropertyDescriptor>() {
            @SuppressWarnings("unchecked")
            @Override
            public boolean apply(final PropertyDescriptor propertyDescriptor) {
                PropertyDescriptor roleProperty = descriptor.getPropertyDescriptor(getRolePropertyName(propertyDescriptor));
                Set<String> requiredRoles = (Set<String>) roleProperty.get(ci);
                return !username.equals(ADMIN) && Sets.intersection(roles, newHashSet(requiredRoles)).isEmpty();
            }
        });
    }

    private static FluentIterable<PropertyDescriptor> getPropertiesWithRolesLimitations(final ConfigurationItem ci, FluentIterable<PropertyDescriptor> properties) {
        return properties.filter(new Predicate<PropertyDescriptor>() {
            public boolean apply(final PropertyDescriptor propertyDescriptor) {
                return ci.hasProperty(getRolePropertyName(propertyDescriptor));
            }
        });
    }

    private static FluentIterable<PropertyDescriptor> getChangedProperties(final Update update, FluentIterable<PropertyDescriptor> properties) {
        return properties.filter(new Predicate<PropertyDescriptor>() {
            public boolean apply(final PropertyDescriptor input) {
                return !input.areEqual(update.getPreviousCi(), update.getNewCi());
            }
        });
    }

    private static FluentIterable<PropertyDescriptor> getSatisfactionProperties(Collection<PropertyDescriptor> properties) {
        return FluentIterable.from(properties).filter(new Predicate<PropertyDescriptor>() {
            public boolean apply(final PropertyDescriptor input) {
                return input.getName().startsWith("satisfies");
            }
        });
    }

    private static String getRolePropertyName(final PropertyDescriptor property) {
        String propName = property.getName().substring("satisfies".length());
        return "roles" + propName;
    }

    @Subscribe(canVeto = true)
    public void checkWhetherUpdateIsAllowed(UpdateCisCommand command) {
        for (Update update : command.getUpdates()) {
            checkUpdate(update, newHashSet(command.getRoles()), command.getUsername());
        }
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 403)
    public static class CannotSetReleaseConditionsException extends DeployitException {
        private CannotSetReleaseConditionsException(List<String> properties, ConfigurationItem ci) {
            super("You are not authorized to set release conditions %s on %s", properties, ci);
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(RepositoryCommandListener.class);
}
