/*
 * @(#)CheckRequiredChangeRequest.java     26 Aug 2011
 *
 * Copyright © 2010 Andrew Phillips.
 *
 * ====================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ====================================================================
 */
package com.xebialabs.deployit.plugins.releaseauth.planning;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableList.Builder;
import com.xebialabs.deployit.plugin.api.deployment.planning.DefaultOrders;
import com.xebialabs.deployit.plugin.api.deployment.planning.PrePlanProcessor;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;


import com.xebialabs.deployit.plugin.api.xld.AppliedDistribution;
import com.xebialabs.deployit.plugin.api.xld.Domain;
import com.xebialabs.deployit.plugins.releaseauth.ConditionVerifier.VerificationResult;
import com.xebialabs.deployit.plugins.releaseauth.ConditionVerifier.ViolatedCondition;
import com.xebialabs.deployit.plugins.releaseauth.ReleaseCondition;
import com.xebialabs.deployit.plugins.releaseauth.step.LogReleaseConditionsStep;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static com.xebialabs.deployit.plugins.releaseauth.ConditionVerifier.validateReleaseConditions;
import static java.lang.Boolean.TRUE;

public class CheckReleaseConditionsAreMet {
    public static final String ENV_RELEASE_CONDITIONS_PROPERTY = "releaseConditions";
    private static final String ENV_RECHECK_CONDITIONS_PROPERTY = "recheckConditionsAtDeploymentTime";
    private static final String ENV_RECHECK_CONDITIONS_ORDER_PROPERTY = "recheckConditionsAtDeploymentTimeOrder";

    private static final List<Step> NO_STEPS = ImmutableList.of();

    @PrePlanProcessor
    public static List<Step> validate(DeltaSpecification spec) {
        AppliedDistribution deployedApplication = spec.getAppliedDistribution();
        Set<ReleaseCondition> conditions = getReleaseConditions(deployedApplication);

        // All fine if there are no conditions
        if (conditions.isEmpty()) {
            return NO_STEPS;
        }

        // Raise error if conditions are not met
        VerificationResult result = validateReleaseConditions(conditions, deployedApplication.getVersion());
        if (result.failed()) {
            throw new IllegalArgumentException(buildErrorMessage(deployedApplication, result.getViolatedConditions()));
        }

        // Add a 'log' step
        return createLogStep(deployedApplication, conditions);
    }

    private static List<Step> createLogStep(AppliedDistribution deployedApplication, Set<ReleaseCondition> conditions) {
        Builder<Step> deploymentSteps = ImmutableList.builder();
        Domain environment = deployedApplication.getEnvironment();
        if (!environment.hasProperty(ENV_RECHECK_CONDITIONS_PROPERTY) || TRUE.equals(environment.getProperty(ENV_RECHECK_CONDITIONS_PROPERTY))) {
            int order = environment.hasProperty(ENV_RECHECK_CONDITIONS_ORDER_PROPERTY) ? environment
                    .<Integer> getProperty(ENV_RECHECK_CONDITIONS_ORDER_PROPERTY) : DefaultOrders.PRE_FLIGHT;
            logger.debug("Adding release auth condition check step at order {}", order);
            deploymentSteps.add(new LogReleaseConditionsStep(order, conditions, deployedApplication.getVersion()));
        }
        return deploymentSteps.build();
    }

    private static Set<ReleaseCondition> getReleaseConditions(AppliedDistribution deployedApplication) {
        final Domain environment = deployedApplication.getEnvironment();

        Set<ReleaseCondition> conditions = new HashSet<>();
        for (PropertyDescriptor property : environment.getType().getDescriptor().getPropertyDescriptors()) {
            Object value = property.get(environment);

            if (property.getName().startsWith("requires") && value != null && (Boolean) value) {
                ReleaseCondition condition = new ReleaseCondition(property.getName().replaceFirst("requires", "satisfies"), property.getLabel());
                conditions.add(condition);
            }
        }

        return conditions;
    }

    private static String buildErrorMessage(AppliedDistribution deployedApplication, Set<ViolatedCondition<?>> violatedConditions) {
        StringBuilder errorMessage = new StringBuilder().append("Cannot deploy '").append(deployedApplication.getName()).append("' (version ")
                .append(deployedApplication.getVersion().getVersion()).append(") to '").append(deployedApplication.getEnvironment().getName())
                .append("' as the following release conditions are not met:");

        for (ViolatedCondition<?> violatedCondition : violatedConditions) {
            errorMessage.append("\n- '").append(violatedCondition.name).append("'");
        }

        return errorMessage.toString();
    }

    private static final Logger logger = LoggerFactory.getLogger(CheckReleaseConditionsAreMet.class);
}
