package com.xebialabs.deployit.plugin.file.deployed;

import com.xebialabs.deployit.plugin.api.deployment.planning.Create;
import com.xebialabs.deployit.plugin.api.deployment.planning.DefaultOrders;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.planning.Modify;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployedArtifact;
import com.xebialabs.deployit.plugin.generic.ci.Container;
import com.xebialabs.deployit.plugin.generic.deployed.CopiedArtifact;
import com.xebialabs.deployit.plugin.overthere.Host;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "Abstract deployed that can target any DeployableArtifact to a Host.")
public class DeployedArtifactOnHost<D extends DeployableArtifact> extends BaseDeployedArtifact<D, Host> {

    @Property(description = "Path to which artifact must be copied to on the host.")
    private String targetPath;

    @Property(required = false, defaultValue = "true", description = "Is the targetPath shared by others on the host. When true, the targetPath is not deleted during undeployment; only the artifacts copied to it.")
    private boolean targetPathShared;

    @Property(required = false, defaultValue = "false", description = "Create the targetPath on the host if it does not exist.")
    private boolean createTargetPath;

    @Property(required = false, description = "Name of the artifact on the host.")
    private String targetFileName;

    @Property(defaultValue = "false", required = false, hidden = true, description = "The sub directories on the target machine are not deleted if files other than that copied by Deployit are present. ")
    private boolean targetPathSharedSubDirectories;


    private CopiedArtifact<D> createDelegate() {
        // Not using the descriptor (which you should always do), because CopiedArtifact is virtual.
        // TODO maybe make a synthetic type which can be used here, which extends CopiedArtifact, just to be used as a delegate.
        CopiedArtifact<D> delegate = new CopiedArtifact<D>(this.getId());
        delegate.setSyntheticProperties(this.getSyntheticProperties());
        delegate.setPlaceholders(this.getPlaceholders());
        delegate.setFile(this.getFile());

        // Not using the descriptor (which you should always do), because Container is virtual.
        // TODO maybe make a synthetic type which can be used here, which extends Container, just to be used as a delegate.
        Container container = new Container(getContainer().getId());
        container.setHost(getContainer());

        delegate.setContainer(container);
        delegate.setDeployable(getDeployable());

        delegate.setCreateOrder(DefaultOrders.CREATE_RESOURCES);
        delegate.setDestroyOrder(DefaultOrders.DESTROY_RESOURCES);

        delegate.setCreateTargetDirectory(isCreateTargetPath());
        delegate.setTargetFile(getTargetFileName());
        delegate.setTargetDirectory(getTargetPath());
        delegate.setTargetDirectoryShared(isTargetPathShared());
        delegate.setUseDescriptionGeneratedByStep(true);
        delegate.setTargetPathSharedSubDirectories(targetPathSharedSubDirectories);

        return delegate;
    }

    @Create
    public void executeCreate(DeploymentPlanningContext ctx, Delta delta) {
        CopiedArtifact<D> delegate = createDelegate();
        delegate.executeCreate(ctx, delta);
    }

    @SuppressWarnings("unchecked")
    @Modify
    public void executeModify(DeploymentPlanningContext ctx, Delta delta) {
        DeployedArtifactOnHost<D> previous = (DeployedArtifactOnHost<D>) delta.getPrevious();
        previous.executeDestroy(ctx, delta);
        executeCreate(ctx, delta);
    }

    @Destroy
    public void executeDestroy(DeploymentPlanningContext ctx, Delta delta) {
        CopiedArtifact<D> delegate = createDelegate();
        delegate.executeDestroy(ctx, delta);
    }

    public String getTargetPath() {
        return targetPath;
    }

    public void setTargetPath(String targetPath) {
        this.targetPath = targetPath;
    }

    public boolean isTargetPathShared() {
        return targetPathShared;
    }

    public void setTargetPathShared(boolean targetPathShared) {
        this.targetPathShared = targetPathShared;
    }

    public boolean isCreateTargetPath() {
        return createTargetPath;
    }

    public void setCreateTargetPath(boolean createTargetPath) {
        this.createTargetPath = createTargetPath;
    }

    public String getTargetFileName() {
        return targetFileName;
    }

    public void setTargetFileName(String targetFileName) {
        this.targetFileName = targetFileName;
    }

}
