package com.xebialabs.deployit.plugin.generic.step;

import static com.google.common.collect.Maps.newHashMap;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Map;

import org.slf4j.MDC;

import com.google.common.base.Preconditions;
import com.xebialabs.deployit.plugin.generic.ci.Container;
import com.xebialabs.deployit.plugin.generic.freemarker.CiAwareObjectWrapper;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.deployit.plugin.generic.freemarker.FileUploader;
import com.xebialabs.deployit.plugin.overthere.ExecutionContextOverthereProcessOutputHandler;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.OverthereProcessOutputHandler;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.overthere.util.OverthereUtils;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;

@SuppressWarnings("serial")
public class ScriptExecutionStep extends GenericBaseStep {

    private static final String MDC_KEY_SCRIPT_PATH = "scriptPath";

	private String scriptTemplatePath;
    private Map<String, Object> vars;

    public ScriptExecutionStep(int order, String scriptPath, Container container, Map<String, Object> vars, String description) {
        super(order,description,container);
        this.scriptTemplatePath = scriptPath;
        this.vars = vars;
        Preconditions.checkNotNull(scriptTemplatePath);
    }

    @Override
    public Result doExecute() throws Exception {
    	MDC.put(MDC_KEY_SCRIPT_PATH, scriptTemplatePath);
    	try {
	        String osSpecificTemplate = resolveOsSpecificTemplate();
	        String executableContent = evaluateTemplate(osSpecificTemplate, vars);
	        logger.debug(executableContent);
	        OverthereFile executable = uploadExecutable(executableContent, substringAfterLast(osSpecificTemplate, '/'));
	        CmdLine cmdLine = CmdLine.build(executable.getPath());
	        OverthereProcessOutputHandler handle = new ExecutionContextOverthereProcessOutputHandler(getCtx());
	        getCtx().logOutput("Executing " + executable.getPath() + " on host " + getContainer().getHost());
	        int rc = getRemoteConnection().execute(handle, cmdLine);
	        if (rc != 0) {
	            getCtx().logError("Execution failed with return code " + rc);
	            return Result.Fail;
	        }
	        return Result.Success;
    	} finally {
    		MDC.remove(MDC_KEY_SCRIPT_PATH);
    	}
    }


    private OverthereFile uploadExecutable(String content, String fileName) {
        OverthereFile targetExecutable = getRemoteWorkingDirectory().getFile(fileName);
        OverthereUtils.write(content.getBytes(), targetExecutable);
        targetExecutable.setExecutable(true);
        return targetExecutable;
    }

    private String evaluateTemplate(String templatePath, Map<String, Object> vars) {
        Configuration cfg = ConfigurationHolder.getConfiguration();
        try {
            Template template = cfg.getTemplate(templatePath);
            StringWriter sw = new StringWriter();
            template.createProcessingEnvironment(vars, sw, new CiAwareObjectWrapper(new WorkingFolderUploader())).process();
            return sw.toString();
        } catch (IOException e) {
            throw new RuntimeIOException(e);
        } catch (TemplateException e) {
            throw new RuntimeException(e);
        }
    }

    private String resolveOsSpecificTemplate() {
        String osSpecificScript = scriptTemplatePath;

        String scriptExt = substringAfterLast(scriptTemplatePath, '.');
        if (scriptExt == null) {
            OperatingSystemFamily os = getContainer().getHost().getOs();
            osSpecificScript = osSpecificScript + os.getScriptExtension();
        }

        if (!classpathResourceExists(osSpecificScript)) {
            throw new IllegalArgumentException("Resource " + osSpecificScript + " not found in classpath");
        }

        return osSpecificScript;
    }

    private boolean classpathResourceExists(String resource) {
        return Thread.currentThread().getContextClassLoader().getResource(resource) != null;
    }

    private class WorkingFolderUploader implements FileUploader {
        private Map<String, String> uploadedFiles = newHashMap();

        @Override
        public String upload(OverthereFile file) {
            if (uploadedFiles.containsKey(file.getName())) {
                return uploadedFiles.get(file.getName());
            }
            OverthereFile uploadedFile = getRemoteWorkingDirectory().getFile(file.getName());
            file.copyTo(uploadedFile);
            uploadedFiles.put(file.getName(), uploadedFile.getPath());
            return uploadedFile.getPath();
        }
    }

    private String substringAfterLast(String str, char sub) {
    	int pos = str.lastIndexOf(sub);
    	if(pos == -1) {
    		return null;
    	} else {
    		return str.substring(pos + 1);
    	}
    }

}
