package com.xebialabs.deployit.plugin.f5;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.generic.ci.Container;
import com.xebialabs.deployit.plugin.generic.ci.NestedContainer;
import com.xebialabs.deployit.plugin.generic.step.ScriptExecutionStep;
import com.xebialabs.deployit.plugin.overthere.HostContainer;

import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Strings.nullToEmpty;

public class F5Contributor {

    @Contributor
    public void f5Activation(Deltas deltas, DeploymentPlanningContext result) {
        Set<Container> targets = gatherTargets(deltas.getDeltas());
        for (Container target : targets) {
            String disableStateScript = getProperty(target, "f5DisableStateScript");
            checkArgument(!nullToEmpty(disableStateScript).trim().isEmpty(), "disable script must be specified.");
            addStep(disableStateScript, getIntProperty(target, "f5DisableOrder"), target, "Disable", result);

            if (notUndeployment(deltas.getDeltas())) {
                String enableStateScript = getProperty(target, "f5EnableStateScript");
                checkArgument(!nullToEmpty(enableStateScript).trim().isEmpty(), "enable script must be specified.");
                addStep(enableStateScript, getIntProperty(target, "f5EnableOrder"), target, "Enable", result);
            }
        }
    }

    private boolean notUndeployment(List<Delta> operations) {
        for (Delta operation : operations) {
            if (operation.getOperation() != Operation.DESTROY) {
                return true;
            }
        }
        return false;
    }

    private String getProperty(Container target, String property) {
        if (hasProperty(target, property)) {
            return (String)target.getProperty(property);
        }
        return "";
    }

    private int getIntProperty(Container target, String property) {
        if (hasProperty(target, property)) {
            return (Integer)target.getProperty(property);
        }
        return 50;
    }
    
    private boolean hasProperty(ConfigurationItem target, String property) {
        Descriptor descriptor = DescriptorRegistry.getDescriptor(target.getType());
        checkArgument(descriptor != null, "Type %s does not exist", target.getType());
        PropertyDescriptor pd = descriptor.getPropertyDescriptor(property);
        return pd != null;
    }


    private void addStep(String script, int scriptOrder, Container container, String verb, DeploymentPlanningContext result) {
        Map<String, Object> vars = Maps.newHashMap();
        vars.put("container", container);
        result.addStep(new ScriptExecutionStep(scriptOrder,script, container, vars, getDescription(verb, container)));
    }

    private String getDescription(String verb, Container container) {
        return String.format("%s F5 node for %s", verb, container.getName());
    }


    private Set<Container> gatherTargets(List<Delta> operations) {
        final Set<Container> targets = Sets.newTreeSet();
        for (Delta operation : operations) {
            addTarget(targets, operation.getOperation(),  operation.getDeployed());
            addTarget(targets, operation.getOperation(), operation.getPrevious());
        }
        return targets;
    }

    private void addTarget(Set<Container> targets, Operation operation, Deployed<?, ?> deployed) {
        if (deployed == null)
            return;

        if (hasProperty(deployed.getContainer(), "f5Enabled") &&
            ((Boolean)deployed.getContainer().getProperty("f5Enabled")) == true) {
            if (operation != Operation.NOOP) {
                targets.add(getRootContainer((HostContainer) deployed.getContainer()));
            }
        }
    }

    private Container getRootContainer(HostContainer hostContainer) {
        if (hostContainer instanceof Container) {
            return (Container)hostContainer;
        } else if (hostContainer instanceof NestedContainer) {
            return ((NestedContainer)hostContainer).getRootContainer();
        } else {
            throw new IllegalStateException("The root container for a nested container not found. NestedContains should be rooted to a generic Container.");
        }
    }
}
