package com.xebialabs.deployit.plugin.generic.ci;

import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionPlanningContext;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseContainer;
import com.xebialabs.deployit.plugin.generic.step.InspectScriptExecutionStep;
import com.xebialabs.deployit.plugin.generic.step.ScriptExecutionStep;
import com.xebialabs.deployit.plugin.overthere.Host;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder.resolveExpression;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "A container that is nested with another container")
public class NestedContainer extends BaseContainer implements GenericContainer {
    private Map<String, Object> freeMarkerContext = Collections.singletonMap("container", (Object) this);

    @Property(required = false, description = "Environment variables for container", label = "Environment Variables")
    private Map<String, String> envVars = newHashMap();

    @Property(required = false, hidden = true, description = "Classpath to the script used to inspect the generic container.")
    private String inspectScript;

    @Property(hidden = true, required = false, description = "Additional classpath resources that should be uploaded to the working directory before executing the inspect script.")
    private Set<String> inspectClasspathResources = newHashSet();

    @Property(hidden = true, required = false, description = "Additional template classpath resources that should be uploaded to the working directory before executing the inspect script." +
            "The template is first rendered and the rendered content copied to a file, with the same name as the template, in the working directory.")
    private Set<String> inspectTemplateClasspathResources = newHashSet();

    public Map<String, String> getEnvVars() {
        return resolveExpression(envVars, freeMarkerContext);
    }

    public void setEnvVars(Map<String, String> envVars) {
        this.envVars = envVars;
    }

    @Inspect
    public void inspectContainer(InspectionPlanningContext ctx) {
        if (emptyToNull(getInspectScript()) != null) {
            InspectScriptExecutionStep step = new InspectScriptExecutionStep(this, getInspectScript(), getHost(), freeMarkerContext, "Inspect " + this);
            step.setTemplateClasspathResources(newArrayList(getInspectTemplateClasspathResources()));
            step.setClasspathResources(newArrayList(getInspectClasspathResources()));
            ctx.addStep(step);
        }
    }


    @SuppressWarnings("rawtypes")
    public List<Step> controlTaskDispatch(String name) {
        String scriptPropertyName = name + "Script";
        PropertyDescriptor propertyDescriptor = DescriptorRegistry.getDescriptor(getType()).getPropertyDescriptor(scriptPropertyName);
        checkArgument(propertyDescriptor != null, "Control task script property %s not defined for CI type %s", scriptPropertyName, getType());
        String scriptName = resolveExpression((String) propertyDescriptor.get(this), freeMarkerContext);

        return Collections.<Step>singletonList(new ScriptExecutionStep(1, scriptName, this,
                freeMarkerContext, "Executing " + name));
    }


    public Container getRootContainer() {
        GenericContainer parentContainer = getParentContainer();
        if (parentContainer instanceof Container) {
            return (Container) parentContainer;
        } else if (parentContainer instanceof NestedContainer) {
            return ((NestedContainer) parentContainer).getRootContainer();
        } else {
            throw new IllegalStateException("The root container for a nested container not found. NestedContains should be rooted to a generic Container.");
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T extends GenericContainer> T getParentContainer() {
        Descriptor descriptor = DescriptorRegistry.getDescriptor(getType());
        for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
            if (pd.isAsContainment()) {
                return (T) pd.get(this);
            }
        }
        throw new IllegalStateException("This nested container " + getType() + " does not have an asContainment relationship to another generic container.");
    }

    @Override
    public Host getHost() {
        return getParentContainer().getHost();
    }

    public String getInspectScript() {
        return resolveExpression(inspectScript, freeMarkerContext);
    }

    public void setInspectScript(String inspectScript) {
        this.inspectScript = inspectScript;
    }

    public Set<String> getInspectClasspathResources() {
        return resolveExpression(inspectClasspathResources, freeMarkerContext);
    }

    public void setInspectClasspathResources(Set<String> inspectClasspathResources) {
        this.inspectClasspathResources = inspectClasspathResources;
    }

    public Set<String> getInspectTemplateClasspathResources() {
        return resolveExpression(inspectTemplateClasspathResources, freeMarkerContext);
    }

    public void setInspectTemplateClasspathResources(Set<String> inspectTemplateClasspathResources) {
        this.inspectTemplateClasspathResources = inspectTemplateClasspathResources;
    }

}
